<?php

/**
 * Elasticsearch PHP Client
 *
 * @link      https://github.com/elastic/elasticsearch-php
 * @copyright Copyright (c) Elasticsearch B.V (https://www.elastic.co)
 * @license   https://opensource.org/licenses/MIT MIT License
 *
 * Licensed to Elasticsearch B.V under one or more agreements.
 * Elasticsearch B.V licenses this file to you under the MIT License.
 * See the LICENSE file in the project root for more information.
 */

declare(strict_types=1);

namespace Elastic\Elasticsearch\Endpoints;

use Elastic\Elasticsearch\Exception\ClientResponseException;
use Elastic\Elasticsearch\Exception\MissingParameterException;
use Elastic\Elasticsearch\Exception\ServerResponseException;
use Elastic\Elasticsearch\Response\Elasticsearch;
use Elastic\Transport\Exception\NoNodeAvailableException;
use Http\Promise\Promise;

/**
 * @generated This file is generated, please do not edit
 */
class Connector extends AbstractEndpoint
{
	/**
	 * Check in a connector
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-check-in
	 * @group serverless
	 * @internal This API is EXPERIMENTAL and may be changed or removed completely in a future release
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function checkIn(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_check_in';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.check_in');
		return $this->client->sendRequest($request);
	}


	/**
	 * Delete a connector
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-delete
	 * @group serverless
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be deleted.
	 *     hard?: bool, // If true, the connector doc is deleted. If false, connector doc is marked as deleted (soft-deleted).
	 *     delete_sync_jobs?: bool, // Determines whether associated sync jobs are also deleted.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function delete(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']);
		$method = 'DELETE';

		$url = $this->addQueryString($url, $params, ['hard','delete_sync_jobs','pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.delete');
		return $this->client->sendRequest($request);
	}


	/**
	 * Get a connector
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-get
	 * @group serverless
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be returned.
	 *     include_deleted?: bool, // A flag indicating whether to return connectors that have been soft-deleted.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function get(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']);
		$method = 'GET';

		$url = $this->addQueryString($url, $params, ['include_deleted','pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.get');
		return $this->client->sendRequest($request);
	}


	/**
	 * Get all connectors
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-list
	 * @group serverless
	 *
	 * @param array{
	 *     from?: int, // Starting offset (default: 0)
	 *     size?: int, // Specifies a max number of results to get (default: 100)
	 *     index_name?: string|array<string>, // A comma-separated list of connector index names to fetch connector documents for
	 *     connector_name?: string|array<string>, // A comma-separated list of connector names to fetch connector documents for
	 *     service_type?: string|array<string>, // A comma-separated list of connector service types to fetch connector documents for
	 *     query?: string, // A search string for querying connectors, filtering results by matching against connector names, descriptions, and index names
	 *     include_deleted?: bool, // A flag indicating whether to return connectors that have been soft-deleted.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 * } $params
	 *
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function list(?array $params = null)
	{
		$params = $params ?? [];
		$url = '/_connector';
		$method = 'GET';

		$url = $this->addQueryString($url, $params, ['from','size','index_name','connector_name','service_type','query','include_deleted','pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, [], $request, 'connector.list');
		return $this->client->sendRequest($request);
	}


	/**
	 * Create a connector
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-put
	 * @group serverless
	 *
	 * @param array{
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body?: string|array<mixed>, // The connector configuration.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function post(?array $params = null)
	{
		$params = $params ?? [];
		$url = '/_connector';
		$method = 'POST';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, [], $request, 'connector.post');
		return $this->client->sendRequest($request);
	}


	/**
	 * Create or update a connector
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-put
	 * @group serverless
	 *
	 * @param array{
	 *     connector_id?: string, // The unique identifier of the connector to be created or updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body?: string|array<mixed>, // The connector configuration.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function put(?array $params = null)
	{
		$params = $params ?? [];
		if (isset($params['connector_id'])) {
			$url = '/_connector/' . $this->encode($params['connector_id']);
			$method = 'PUT';
		} else {
			$url = '/_connector';
			$method = 'PUT';
		}
		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.put');
		return $this->client->sendRequest($request);
	}


	/**
	 * Cancel a connector sync job
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-sync-job-cancel
	 * @group serverless
	 *
	 * @param array{
	 *     connector_sync_job_id: string, // (REQUIRED) The unique identifier of the connector sync job to be canceled
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function syncJobCancel(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_sync_job_id'], $params);
		$url = '/_connector/_sync_job/' . $this->encode($params['connector_sync_job_id']) . '/_cancel';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_sync_job_id'], $request, 'connector.sync_job_cancel');
		return $this->client->sendRequest($request);
	}


	/**
	 * Check in a connector sync job
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-sync-job-check-in
	 * @internal This API is EXPERIMENTAL and may be changed or removed completely in a future release
	 *
	 * @param array{
	 *     connector_sync_job_id: string, // (REQUIRED) The unique identifier of the connector sync job to be checked in
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function syncJobCheckIn(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_sync_job_id'], $params);
		$url = '/_connector/_sync_job/' . $this->encode($params['connector_sync_job_id']) . '/_check_in';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_sync_job_id'], $request, 'connector.sync_job_check_in');
		return $this->client->sendRequest($request);
	}


	/**
	 * Claim a connector sync job
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-sync-job-claim
	 * @internal This API is EXPERIMENTAL and may be changed or removed completely in a future release
	 *
	 * @param array{
	 *     connector_sync_job_id: string, // (REQUIRED) The unique identifier of the connector sync job to be claimed.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) Data to claim a sync job.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function syncJobClaim(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_sync_job_id','body'], $params);
		$url = '/_connector/_sync_job/' . $this->encode($params['connector_sync_job_id']) . '/_claim';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_sync_job_id'], $request, 'connector.sync_job_claim');
		return $this->client->sendRequest($request);
	}


	/**
	 * Delete a connector sync job
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-sync-job-delete
	 * @group serverless
	 *
	 * @param array{
	 *     connector_sync_job_id: string, // (REQUIRED) The unique identifier of the connector sync job to be deleted.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function syncJobDelete(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_sync_job_id'], $params);
		$url = '/_connector/_sync_job/' . $this->encode($params['connector_sync_job_id']);
		$method = 'DELETE';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_sync_job_id'], $request, 'connector.sync_job_delete');
		return $this->client->sendRequest($request);
	}


	/**
	 * Set a connector sync job error
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-sync-job-error
	 * @internal This API is EXPERIMENTAL and may be changed or removed completely in a future release
	 *
	 * @param array{
	 *     connector_sync_job_id: string, // (REQUIRED) The unique identifier of the connector sync job to set an error for.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) The error to set in the connector sync job.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function syncJobError(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_sync_job_id','body'], $params);
		$url = '/_connector/_sync_job/' . $this->encode($params['connector_sync_job_id']) . '/_error';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_sync_job_id'], $request, 'connector.sync_job_error');
		return $this->client->sendRequest($request);
	}


	/**
	 * Get a connector sync job
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-sync-job-get
	 * @group serverless
	 *
	 * @param array{
	 *     connector_sync_job_id: string, // (REQUIRED) The unique identifier of the connector sync job to be returned.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function syncJobGet(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_sync_job_id'], $params);
		$url = '/_connector/_sync_job/' . $this->encode($params['connector_sync_job_id']);
		$method = 'GET';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_sync_job_id'], $request, 'connector.sync_job_get');
		return $this->client->sendRequest($request);
	}


	/**
	 * Get all connector sync jobs
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-sync-job-list
	 * @group serverless
	 *
	 * @param array{
	 *     from?: int, // Starting offset (default: 0)
	 *     size?: int, // specifies a max number of results to get (default: 100)
	 *     status?: string, // A sync job status to fetch connector sync jobs for
	 *     connector_id?: string, // Id of the connector to fetch the sync jobs for
	 *     job_type?: string|array<string>, // A comma-separated list of job types
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 * } $params
	 *
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function syncJobList(?array $params = null)
	{
		$params = $params ?? [];
		$url = '/_connector/_sync_job';
		$method = 'GET';

		$url = $this->addQueryString($url, $params, ['from','size','status','connector_id','job_type','pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, [], $request, 'connector.sync_job_list');
		return $this->client->sendRequest($request);
	}


	/**
	 * Create a connector sync job
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-sync-job-post
	 * @group serverless
	 *
	 * @param array{
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) The connector sync job data.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function syncJobPost(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['body'], $params);
		$url = '/_connector/_sync_job';
		$method = 'POST';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, [], $request, 'connector.sync_job_post');
		return $this->client->sendRequest($request);
	}


	/**
	 * Set the connector sync job stats
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-sync-job-update-stats
	 * @internal This API is EXPERIMENTAL and may be changed or removed completely in a future release
	 *
	 * @param array{
	 *     connector_sync_job_id: string, // (REQUIRED) The unique identifier of the connector sync job to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) The stats to update for the connector sync job.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function syncJobUpdateStats(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_sync_job_id','body'], $params);
		$url = '/_connector/_sync_job/' . $this->encode($params['connector_sync_job_id']) . '/_stats';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_sync_job_id'], $request, 'connector.sync_job_update_stats');
		return $this->client->sendRequest($request);
	}


	/**
	 * Activate the connector draft filter
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-update-filtering
	 * @group serverless
	 * @internal This API is EXPERIMENTAL and may be changed or removed completely in a future release
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function updateActiveFiltering(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_filtering/_activate';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.update_active_filtering');
		return $this->client->sendRequest($request);
	}


	/**
	 * Update the connector API key ID
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-update-api-key-id
	 * @group serverless
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) An object containing the connector's API key id and/or Connector Secret document id for that API key.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function updateApiKeyId(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id','body'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_api_key_id';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.update_api_key_id');
		return $this->client->sendRequest($request);
	}


	/**
	 * Update the connector configuration
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-update-configuration
	 * @group serverless
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) Mapping between field names to configuration.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function updateConfiguration(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id','body'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_configuration';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.update_configuration');
		return $this->client->sendRequest($request);
	}


	/**
	 * Update the connector error field
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-update-error
	 * @group serverless
	 * @internal This API is EXPERIMENTAL and may be changed or removed completely in a future release
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) An object containing the connector's error.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function updateError(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id','body'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_error';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.update_error');
		return $this->client->sendRequest($request);
	}


	/**
	 * Update the connector features
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-update-features
	 * @internal This API is EXPERIMENTAL and may be changed or removed completely in a future release
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) An object containing the connector's features definition.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function updateFeatures(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id','body'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_features';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.update_features');
		return $this->client->sendRequest($request);
	}


	/**
	 * Update the connector filtering
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-update-filtering
	 * @group serverless
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) A list of connector filtering configurations.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function updateFiltering(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id','body'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_filtering';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.update_filtering');
		return $this->client->sendRequest($request);
	}


	/**
	 * Update the connector draft filtering validation
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-update-filtering-validation
	 * @group serverless
	 * @internal This API is EXPERIMENTAL and may be changed or removed completely in a future release
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) Validation info for the draft filtering rules. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function updateFilteringValidation(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id','body'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_filtering/_validation';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.update_filtering_validation');
		return $this->client->sendRequest($request);
	}


	/**
	 * Update the connector index name
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-update-index-name
	 * @group serverless
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) An object containing the connector's index name.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function updateIndexName(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id','body'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_index_name';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.update_index_name');
		return $this->client->sendRequest($request);
	}


	/**
	 * Update the connector name and description
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-update-name
	 * @group serverless
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) An object containing the connector's name and/or description.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function updateName(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id','body'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_name';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.update_name');
		return $this->client->sendRequest($request);
	}


	/**
	 * Update the connector is_native flag
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-update-native
	 * @group serverless
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) An object containing the connector's is_native flag. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function updateNative(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id','body'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_native';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.update_native');
		return $this->client->sendRequest($request);
	}


	/**
	 * Update the connector pipeline
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-update-pipeline
	 * @group serverless
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) An object with connector ingest pipeline configuration.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function updatePipeline(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id','body'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_pipeline';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.update_pipeline');
		return $this->client->sendRequest($request);
	}


	/**
	 * Update the connector scheduling
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-update-scheduling
	 * @group serverless
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) An object containing the connector's scheduling configuration.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function updateScheduling(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id','body'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_scheduling';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.update_scheduling');
		return $this->client->sendRequest($request);
	}


	/**
	 * Update the connector service type
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-update-service-type
	 * @group serverless
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) An object containing the connector's service type.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function updateServiceType(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id','body'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_service_type';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.update_service_type');
		return $this->client->sendRequest($request);
	}


	/**
	 * Update the connector status
	 *
	 * @link https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-connector-update-status
	 * @group serverless
	 * @internal This API is EXPERIMENTAL and may be changed or removed completely in a future release
	 *
	 * @param array{
	 *     connector_id: string, // (REQUIRED) The unique identifier of the connector to be updated.
	 *     pretty?: bool, // Pretty format the returned JSON response. (DEFAULT: false)
	 *     human?: bool, // Return human readable values for statistics. (DEFAULT: true)
	 *     error_trace?: bool, // Include the stack trace of returned errors. (DEFAULT: false)
	 *     source?: string, // The URL-encoded request definition. Useful for libraries that do not accept a request body for non-POST requests.
	 *     filter_path?: string|array<string>, // A comma-separated list of filters used to reduce the response.
	 *     body: string|array<mixed>, // (REQUIRED) An object containing the connector's status.. If body is a string must be a valid JSON.
	 * } $params
	 *
	 * @throws MissingParameterException if a required parameter is missing
	 * @throws NoNodeAvailableException if all the hosts are offline
	 * @throws ClientResponseException if the status code of response is 4xx
	 * @throws ServerResponseException if the status code of response is 5xx
	 *
	 * @return Elasticsearch|Promise
	 */
	public function updateStatus(?array $params = null)
	{
		$params = $params ?? [];
		$this->checkRequiredParameters(['connector_id','body'], $params);
		$url = '/_connector/' . $this->encode($params['connector_id']) . '/_status';
		$method = 'PUT';

		$url = $this->addQueryString($url, $params, ['pretty','human','error_trace','source','filter_path']);
		$headers = [
			'Accept' => 'application/json',
			'Content-Type' => 'application/json',
		];
		$request = $this->createRequest($method, $url, $headers, $params['body'] ?? null);
		$request = $this->addOtelAttributes($params, ['connector_id'], $request, 'connector.update_status');
		return $this->client->sendRequest($request);
	}
}
