<?php
namespace Teknisa\Libs\Service;

use Teknisa\Libs\Util\Utilities;
use Teknisa\Libs\Util\InstanceProvider;
use Zeedhi\Framework\Session\Session;
use Teknisa\Libs\Util\Login\UserData;
use Teknisa\Libs\Exception\Login as LoginException;

class Api {
    /** @var Session $sessionManager */
	protected $sessionManager;
    /** @var UserData $userData */
    private $userData;
    
    protected $config;

    const BIRT_EXTENSION = 'rptdesign';
    const CONNECTION_JS_FILE_NAME = 'connection.js';

    const DEFAULT_TRANSLATE_URL   = 'https://translate.teknisa.com/backend/index.php';
    const DEFAULT_FILE_NAME       = 'translation';
    const DEFAULT_WORDS_PATH      = 'frontend/public/metadata/json/words';
    const DEFAULT_DIST_WORDS_PATH = 'frontend/dist/metadata/json/words';

    /**
     * @param Session $sessionManager
     * @param UserData $userData
     */
    public function __construct(Session $sessionManager, UserData $userData) {
        $this->sessionManager = $sessionManager;
        $this->userData = $userData;
        $this->config = json_decode(file_get_contents(__DIR__ . '/../../../../config/config.json'), true);
    }

    /**
     * @param $nrOrg
     * @param $cdOperator
     */
    public function setSession($nrOrg, $cdOperator)
    {
        $this->sessionManager->set('NRORG', $nrOrg);
        $this->sessionManager->set('CDOPERADOR', $cdOperator);
    }

    /**
     * @param $user
     * @param $password
     * @return array
     */
    public function getUserData($user, $password) {
        return $this->userData->getUserData($user, $password);
    }

    /**
     * @param $email
     * @param $password
     * @param $identification
     * @return array
     */
    public function updatePassword($email, $password, $identification) {
        $this->userData->updatePassword($email, $password, $identification);
    }

    /**
     * @param $user
     * @param $password
     * @return array
     */
    public function validateUser($user, $password) {
        return $this->userData->validateUser($user, $password);
    }

    /**
     * @return array
     */
    public function getModulesVersion() {
        $productBasePath = Utilities::getProductBasePath();
        $modules = @file_get_contents($productBasePath . 'modules.json');
        if (!$modules) {
            LoginException::modulesConfigNotFound();
        }

        $modules = json_decode($modules, true);
        $modulesVersion = $modulesParametersNotFound = [];
        foreach($modules as $key => $module) {
            if(!isset($module['onlyControl'])) {
                if($key == Utilities::PRODUCT_MENU_NAME) {
                    $versionFile = @file_get_contents($productBasePath . 'frontend/package.json');
					if(!$versionFile) {
						$versionFile = @file_get_contents($productBasePath . 'mobile/package.json');
					}
                } else {
                    $versionFile = @file_get_contents($productBasePath . 'modules/' . $key . '/frontend/package.json');
					if(!$versionFile) {
						$versionFile = @file_get_contents($productBasePath . 'modules/' . $key . '/mobile/package.json');
					}					
                }
                if((!isset($module['maxVersion']) && $key != Utilities::PRODUCT_MENU_NAME) || !isset($module['devOpsId'])) {
                    $modulesParametersNotFound[] = $key;
                } else {
                    $versionFile = json_decode($versionFile, true);
                    $modulesVersion[] = array(
                        'KEY'         => $key,
                        'NAME'        => isset($versionFile['name']) ? $versionFile['name'] : '',
                        'VERSION'     => !$versionFile ? '-' : $versionFile['version'],
                        'MAX_VERSION' => $key != Utilities::PRODUCT_MENU_NAME ? $module['maxVersion'] : '*',
                        'DEVOPS_ID'   => $module['devOpsId']
                    );
                }
            }
        }

        if (!empty($modulesParametersNotFound)) {
            LoginException::moduleParametersNotFound(join('",<br/>"', $modulesParametersNotFound));
        }
        return $modulesVersion;
    }

    /**
     * @return string
     */
    public function generateModulesVersionFile() {
        $modulesVersion = $this->getModulesVersion();
        $modulesVersion = json_encode($modulesVersion);
        $generatedFile  = Utilities::getGeneratedFile();
        file_put_contents($generatedFile, $modulesVersion);
        $generatedFile = explode('/', $generatedFile);
        return array_pop($generatedFile);
    }

    public function downloadLoginAccessFile($fileName) {
        Utilities::downloadAndRemoveFile(Utilities::getGeneratedFileDir() . '/' . $fileName, $fileName);
    }

    public function setLastRequestTime($route) {
        if($this->sessionManager->isStarted() && $route != '/lib_validateExpiredSession') {
            $date = new \DateTime();
            $this->sessionManager->set('LAST_REQUEST_TIME', $date->getTimestamp());
        }
    }

    public function updateTranslation() {
        $ds = DIRECTORY_SEPARATOR;
        $productBasePath = Utilities::getProductBasePath();
        $products = $this->getProductsTranslation($productBasePath);
        $response = $this->callWordsWebService($products);

        foreach($products as $module) {
            $productId = $module['id'];
            if(!empty($response[$productId])) {
                $moduleResponse = $response[$productId];
                $this->buildWordFiles($moduleResponse['json'], $productBasePath, $ds, $module);
                if(!empty($moduleResponse['repository']) && $moduleResponse['repository']['NEW_STRUCTURE'] == 'Y') {
                    $this->buildBirtFiles($moduleResponse['birt'], $moduleResponse['repository'], $productBasePath, $ds, $module);
                }
            }
        }
    }

    protected function buildWordFiles($moduleJson, $productBasePath, $ds, $module) {
        $defaultPath = is_dir($productBasePath . self::DEFAULT_DIST_WORDS_PATH) ? self::DEFAULT_DIST_WORDS_PATH : self::DEFAULT_WORDS_PATH; 
        $wordsPath = !empty($module['wordsPath']) ? $module['wordsPath'] : $defaultPath;
        $path = $this->getWordsBasePath($productBasePath, $wordsPath, $ds, $module);
        if($path) {
            $fileData = array();
            foreach($moduleJson as $word) {
                $wordValue = $word['WORDS_VALUE'];
                if(!is_numeric($wordValue) && !empty(trim($wordValue))) {
                    $fileData[strtolower($word['LANGUAGE_ISO'])][$wordValue] = $word['VALUE'];
                }
            }
            
            foreach($fileData as $fileName => $fileTranslation) {
                $filePath = $path . $ds . $fileName . '.json';
                file_put_contents($filePath, json_encode($fileTranslation, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT));
            }
        }
    }

    protected function buildBirtFiles($moduleBirt, $repository, $productBasePath, $ds, $module) {
        $wordsPath = $repository['WORDS_PATH'];
        $messagesJson = $repository['MESSAGES_FILE'];
        if(!empty($module['birtBasePath'])) {
            $wordsPath  = !empty($module['birtWordsPath']) ? $module['birtWordsPath'] : $module['birtBasePath'] . $ds . $repository['WORDS_PATH'];
            if(!empty($module['birtMessagesJson']) || !empty($repository['MESSAGES_FILE'])) {
                $messagesJson = !empty($module['birtMessagesJson']) ? $module['birtMessagesJson'] : $module['birtBasePath'] . $ds . $repository['MESSAGES_FILE'];
            }
        }

        $path = $this->getWordsBasePath($productBasePath, $wordsPath, $ds, $module);
        if($path) {
            $defaultLangKeys = $this->getDefaultLangToUseKeys($moduleBirt);
            $fileData = array();
            $fileTranslationData = '';
            foreach($moduleBirt as $word) {
                $wordValue = $word['WORDS_VALUE'];
                if(!is_numeric($wordValue) && !empty(trim($wordValue))) {
                    $fileData[$word['LANGUAGE_ISO']]  = empty($fileData[$word['LANGUAGE_ISO']]) ? '' : $fileData[$word['LANGUAGE_ISO']];
                    $wordValueEsc = $this->unicodeEscape($wordValue);
                    $fileData[$word['LANGUAGE_ISO']] .= $wordValueEsc . '=' . $this->unicodeEscape($word['VALUE']) . PHP_EOL;
                    if($word['LANGUAGE_ISO'] == $defaultLangKeys) {
                        $fileTranslationData .= $wordValueEsc . '=' . $wordValueEsc . PHP_EOL;
                    }
                }
            }

            foreach($fileData as $langIso => $fileTranslation) {
                $filePath = $path . $ds . self::DEFAULT_FILE_NAME . '_' . $langIso . '.properties';
                file_put_contents($filePath, $fileTranslation);
            }
            $fileTranslationData .= $this->getMessagesJson($productBasePath, $messagesJson, $ds, $repository, $module);
            $fileTranslationPath = $path . $ds . self::DEFAULT_FILE_NAME . '.properties';
            file_put_contents($fileTranslationPath, $fileTranslationData);
        }
    }

    protected function getMessagesJson($productBasePath, $messagesJson, $ds, $repository, $module) {
        $data = '';
        $path = $this->getModuleBasePath($productBasePath, $module, $ds);
        $fileContents = @file_get_contents($path . $ds . $messagesJson);
        if(!empty($fileContents)) {
            $messages = @json_decode($fileContents, true);
            if(!empty($messages)) {
                foreach($messages[$repository['MESSAGES_STRUCTURE']] as $key => $value) {
                    $data .= $this->unicodeEscape($key) . '=' . $this->unicodeEscape($value) . PHP_EOL;
                }
            }
        }
        return $data;
    }

    protected function getWordsBasePath($productBasePath, $wordsPath, $ds, $module) {
        $path = $this->getModuleBasePath($productBasePath, $module, $ds);
        if(!is_dir($path)) {
            return false;
        }
        $path .= $ds . $wordsPath;
        if(!is_dir($path)) {
            mkdir($path, 0777, true);
        }
        return $path;
    }

    protected function getModuleBasePath($productBasePath, $module, $ds) {
        return $productBasePath . ($module['path'] == 'product' ? '' : $ds . 'modules' . $ds . $module['path']);
    }

    protected function unicodeEscape($string) {
        $newStr = '';
        $regex = '/[^0-9|^a-z]/i';
        $chars = $this->strSplitUnicode($string);
        foreach($chars as $char) {
            if(preg_match($regex, $char)) {
                $enconding = mb_convert_encoding($char, 'UTF-32BE', 'UTF-8');
                $newStr .= "\u" . substr(bin2hex($enconding), -4);
            } else {
                $newStr .= $char;
            }
        }
        return $newStr;
    }

    protected function strSplitUnicode($str, $l = 1) {
        if ($l > 0) {
            $ret = array();
            $len = mb_strlen($str, "UTF-8");
            for ($i = 0; $i < $len; $i += $l) {
                $ret[] = mb_substr($str, $i, $l, "UTF-8");
            }
            return $ret;
        }
        return preg_split("//u", $str, -1, PREG_SPLIT_NO_EMPTY);
    }

    protected function getDefaultLangToUseKeys($moduleBirt) {
        foreach($moduleBirt as $word) {
            if($word['LANGUAGE_ISO'] == 'es_CL' || $word['LANGUAGE_ISO'] == 'en_US') {
                return $word['LANGUAGE_ISO'];
            }
        }
        return 'en_US';
    }

    protected function getProductsTranslation($productBasePath) {
        $products = array();
        $modules = @file_get_contents($productBasePath . $this->config['modulesPath']);
        if (!$modules) {
            LoginException::modulesConfigNotFound();
        }
        $modules = json_decode($modules, true);
        foreach($modules as $key => $module) {            
            if(empty($module['onlyControl']) && empty($module['nextProduct']) && empty($module['nextModule'])) {
                $currentModule = $module;
                $currentModule['id'] = $module['devOpsId'];
                $currentModule['path'] = $key;
                $currentModule['repositoryId'] = !empty($module['repositoryId']) ? $module['repositoryId'] : null;
                $products[] = $currentModule;
            }
        }
        return $products;
    }

    protected function callWordsWebService($products) {
        $url = strpos($this->config['translateUrl'], '[TRANSLATE_URL]') !== false ? self::DEFAULT_TRANSLATE_URL : $this->config['translateUrl'];
        $params = array(
            "requestType" => "Row",
            "row" => array(
                'products' => $products,
                'organizationId' => 20,
            )
        );
        return Utilities::callWebService(null, '/getWordsByModules', $params, 'getWordsByModules', $this->config['loginClientSecret'], $url);
    }

    public function openReport($key)
    {
        $filter = array(
            'KEY' => $key
        );
        $reportData = InstanceProvider::getMongoDB()->find('ZhReportData', $filter);

        if(empty($reportData)) {
            echo '<h3>Relatório não encontrado. <br> Utilize o sistema para gerar novamente.<h3>';
            die;
        } else {
            $format = !empty($reportData[0]['FORMAT']) ? $reportData[0]['FORMAT'] : '';
            $url    = $reportData[0]['URL'];
            
            if(strtolower($format) == 'html') {
                echo file_get_contents($url);
            } else {
                $name = $reportData[0]['NAME'] . '.pdf';
                header('Accept-Ranges: none');
                header("Content-Type: application/pdf");
                header("Content-Disposition: inline; filename=" . $name);
                readfile($url);
            }
            InstanceProvider::getMongoDB()->remove('ZhReportData', $filter);
            exit;
        }
    }

    public function getReportContent($fileName, $isCustom) {
        $ds = DIRECTORY_SEPARATOR;
        if($isCustom) {
            $nrorg = InstanceProvider::getEnvironment()->getNrorg();
            $productId = Utilities::getProjectIdParameter();
            $basePath = self::getCustomReportsPath();
        } else {
            $birtStrategy = InstanceProvider::getBirtStrategy();
            $basePath = $birtStrategy->getReportPath();
        }

        $filePath = $basePath . $ds . $fileName . '.' . self::BIRT_EXTENSION;
        return array (
            'extension' => self::BIRT_EXTENSION,
            'content'   => base64_encode(file_get_contents($filePath))
        );
    }

    public function createCustomReport($params) {
        $customReportsRepository = InstanceProvider::getLibCustomReportsRepository();
        $nrorg = InstanceProvider::getEnvironment()->getNrorg();
        $productId = Utilities::getProjectIdParameter();
        $isCustom = isset($params['REPORT_MODAL']['isCustom']) ? $params['REPORT_MODAL']['isCustom'] : false;
        $isUpdate = $isCustom && $params['OPERATION'] == 'U';

        if($isUpdate) {
            $fileName = $params['REPORT_MODAL']['value'];
            $id = $params['REPORT_MODAL']['id'];
            $customReport = $customReportsRepository->getCustomReports($productId, $params['WINDOW_NAME'], $nrorg, $id);
            $parentReport = $customReport[0]['PARENT_REPORT'];
        } else {
            if($isCustom) {
                $id = $params['REPORT_MODAL']['id'];
                $customReport = $customReportsRepository->getCustomReports($productId, $params['WINDOW_NAME'], $nrorg, $id);
                $parentReport = $customReport[0]['PARENT_REPORT'];
            } else {
                $parentReport = $params['REPORT_MODAL']['value'];
            }
            $fileName = uniqid($parentReport . '_');
        }

        $this->createCustomReportFile($params, $nrorg, $productId, $fileName, $parentReport);
        $this->saveCustomReportDb($params, $nrorg, $productId, $fileName, $isCustom, $isUpdate, $parentReport);

        return $isUpdate ? 'Update' : 'New';
    }

    private function saveCustomReportDb($params, $nrorg, $productId, $fileName, $isCustom, $isUpdate, $parentReport) {
        $customReportsRepository = InstanceProvider::getLibCustomReportsRepository();
        $cdoperador = InstanceProvider::getEnvironment()->getCdoperador();
        $result = 'New';
        if($isUpdate) {
            $id = $params['REPORT_MODAL']['id'];
            $customReportsRepository->updateCustomReport($id, $params['NAME'], $productId,
                $params['WINDOW_NAME'], $nrorg, $cdoperador);
            $result = 'Update';
        } else {
            $customReportsRepository->insertCustomReport($params['NAME'], $fileName, $parentReport,
                $productId, $params['WINDOW_NAME'], $nrorg, $cdoperador);
        }
        return $result;
    }

    private function createCustomReportFile($params, $nrorg, $productId, $fileName, $parentReport) {
        $content = base64_decode(str_replace('data:application/octet-stream;base64,', '', $params['FILE_CONTENT']));
        $this->validateReportContent($content, $parentReport);
        $ds = DIRECTORY_SEPARATOR;
        $productPath = self::getCustomReportsPath();
        if(!is_dir($productPath)) {
            mkdir($productPath, 0777, true);
        }
        $file = $productPath . $ds . $fileName . '.' . self::BIRT_EXTENSION;
        file_put_contents($file, $content);
    }

    private function validateReportContent($content, $parentReport) {
        $birtStrategy = InstanceProvider::getBirtStrategy();
        $parentReportPath = $birtStrategy->getReportPath() . DIRECTORY_SEPARATOR . $parentReport . '.' . self::BIRT_EXTENSION;
        $parentXml = json_decode(json_encode(simplexml_load_file($parentReportPath)), true);
        $xml = json_decode(json_encode(simplexml_load_string($content)), true);
        $this->validateReportParams($xml, $parentXml);
        $this->validateReportQueries($content, $xml);
    }

    private function validateReportParams($xml, $parentXml) {
        $parentReportParams = $reportParams = array();

        if(!empty($parentXml['parameters']) && !empty($parentXml['parameters']['scalar-parameter'])) {
            foreach($parentXml['parameters']['scalar-parameter'] as $parameter) {
                $parentReportParams[] = $parameter['@attributes']['name'];
            }
        }

        if(!empty($xml['parameters']) && !empty($xml['parameters']['scalar-parameter'])) {
            foreach($xml['parameters']['scalar-parameter'] as $parameter) {
                $reportParams[] = $parameter['@attributes']['name'];
            }
        }

        if(!empty(array_diff($reportParams, $parentReportParams))) {
            throw new \Exception('LIB_REPORT_INVALID_PARAMS');
        }
    }

    private function validateReportQueries($content, $xml) {
        if(!empty($xml['oda-data-set']) && !empty($xml['data-sets']['oda-data-set'])) {
            $upperContent = strtoupper($content);
            $validation = array();
            $invalidCommands = array (
                'CREATE ', 'UPDATE ', 'ALTER ', 'TRUNCATE ',
                'DROP ', 'INSERT ', 'GRANT  ', 'COMMIT ', 'REVOKE '
            );

            foreach($invalidCommands as $command) {
                if(strpos($upperContent, $command) !== false) {
                    $validation[] = $command;
                }
            }

            if(!empty($validation)) {
                throw new \Exception('LIB_REPORT_INVALID_QUERIES');
            }
        }
    }

    public function getCustomReports($windowName) {
        $nrorg = InstanceProvider::getEnvironment()->getNrorg();
        $productId = Utilities::getProjectIdParameter();
        return InstanceProvider::getLibCustomReportsRepository()->getCustomReports($productId, $windowName, $nrorg);
    }

    public function deleteCustomReport($params) {
        $nrorg = InstanceProvider::getEnvironment()->getNrorg();
        $productId = Utilities::getProjectIdParameter();
        InstanceProvider::getLibCustomReportsRepository()->deleteCustomReport($params['ID'], $productId, $params['WINDOW_NAME'], $nrorg);
        $this->deleteCustomReportFile($params);
    }

    private function deleteCustomReportFile($params) {
        $ds = DIRECTORY_SEPARATOR;
        $file = self::getCustomReportsPath() . $ds . $params['FILE_NAME'] . '.' . self::BIRT_EXTENSION;
        @unlink($file);
    }

    public static function getCustomReportsPath() {
        $productId = Utilities::getProjectIdParameter();
        $nrorg = InstanceProvider::getEnvironment()->getNrorg();
        $ds = DIRECTORY_SEPARATOR;
        if(strpos(__DIR__, 'modules') === false) {
            $modulePath = 'product';
        } else {
            $dirPath = explode('modules/', __DIR__)[1];
            $modulePath = explode('/', $dirPath)[0];
        }
        return Utilities::getCustomizationsPathParameter() . $ds . $nrorg . $ds . $productId . $ds . $modulePath;
    }

    public static function copyConnectionJs($actualPath = null, $destinyPath = null) {
        $birtStrategy = InstanceProvider::getBirtStrategy();
        $actualPath  = $actualPath  ?: $birtStrategy->getReportPath();
        $destinyPath = $destinyPath ?: self::getCustomReportsPath();

        $actualFile  = $actualPath  . DIRECTORY_SEPARATOR . self::CONNECTION_JS_FILE_NAME;
        $destinyFile = $destinyPath . DIRECTORY_SEPARATOR . self::CONNECTION_JS_FILE_NAME;
        
        if(file_exists($destinyFile)) {
            if(filesize($destinyFile) !== filesize($actualFile)) {
                unlink($destinyFile);
                copy($actualFile, $destinyFile);
            }
        } else {
            copy($actualFile, $destinyFile);
        }

        self::copyWordFiles($destinyPath);
    }

    private static function copyWordFiles($destinyPath) {
        $wordsPathParameter = Utilities::getWordsPathParameter();
        $wordsPath = $wordsPathParameter ? Utilities::getProductBasePath() . DIRECTORY_SEPARATOR . $wordsPathParameter : null;
        if($wordsPath && is_dir($wordsPath)) {
            $files = array_diff(scandir($wordsPath), array('.', '..'));
            $destinyWordsPath = $destinyPath . DIRECTORY_SEPARATOR . 'words';
            if(!is_dir($destinyWordsPath)) {
                mkdir($destinyWordsPath, 0777, true);
            }
            foreach($files as $file) {
                $filePath = $wordsPath . DIRECTORY_SEPARATOR . $file;
                if(is_file($filePath)) {
                    $destinyFilePath = $destinyWordsPath . DIRECTORY_SEPARATOR . $file;
                    if(file_exists($destinyFilePath)) {
                        if(filesize($destinyFilePath) !== filesize($filePath)) {
                            unlink($destinyFilePath);
                            copy($filePath, $destinyFilePath);
                        }
                    } else {
                        copy($filePath, $destinyFilePath);
                    }
                }
            }
        }
    }

    public function chatBotQuery($sql, $sqlParams) {
        return InstanceProvider::getEntityManager()->getConnection()->fetchAll($sql, $sqlParams);
    }
}