<?php

namespace Teknisa\Libs\Traits;

use Doctrine\ORM\EntityManager;

trait TQuery {

    /**
     * @return string|null
     */
    private function getPage()
    {
        return $_REQUEST['page'] ?? 1;
    }

    /**
     * @return string|null
     */
    private function getLimit(): ?string
    {
        return $_REQUEST['limit'] ?? null;
    }

    /**
     * @return string|null
     */
    private function getOrder(): ?array
    {
        return $_REQUEST['order'] ?? null;
    }

    /**
     * @return array|null
     */
    private function getSearch(): ?string
    {
        return $_REQUEST['search'] ?? null;
    }

    /**
     * @return array|null
     */
    private function getSearchIn(): ?array
    {
        return $_REQUEST['search_in'] ?? null;
    }

    /**
     * @param $query
     * @param string $pagination
     * @param string $orderBy
     * @param string $search
     * @return string
     */
    private function buildQuery($query, $pagination = '', $orderBy = '', $search = ''): string
    {
        return "SELECT * 
                 FROM ( SELECT USER_QUERY.*, ROWNUM RNUM, COUNT(1) OVER () TOTAL_LINES
                          FROM ( $query ) USER_QUERY 
                       $search)
                $pagination
                $orderBy";
    }

    /**
     * @param $page
     * @param $limit
     * @return string
     */
    public function createPagination ($page, $limit): string
    {
        if($page && $limit) {
            $end = $page * $limit;
            $init = $end - $limit + 1;
            return "WHERE RNUM BETWEEN $init AND $end";
        }
        return "";
    }

    /**
     * @param $search
     * @param $searchIn
     * @return string
     *
     */
    public function createSearch ($search, $searchIn): string
    {
        if (is_array($searchIn)) {
            $where = 'WHERE ';
            end($searchIn);         // move the internal pointer to the end of the array
            $lastKey = key($searchIn);

            foreach ($searchIn as $index => $column) {
                $where .= "(UPPER(\"$column\") LIKE UPPER('%$search%'))";
                if ($lastKey != $index) {
                    $where .= " OR ";
                }
            }
        } else {
            $where = "WHERE (UPPER(\"$searchIn\") LIKE UPPER('%$search%'))";
        }
        return $where;
    }

    /**
     * @param $orderBy
     * @return string
     */
    public function createOrderBy ($orderBy): string
    {
        $where = "ORDER BY ";

        if (is_array($orderBy)) {
            $where .= implode(", ", str_replace('.', ' ', $orderBy));
        } else {
            $where .= str_replace('.', ' ', $orderBy);
        }

        return strtoupper($where);
    }

    /**
     * @param $data
     * @param $page
     * @param $limit
     * @return array
     */
    public function buildResponseData($data, $page = null, $limit = null): array
    {
        if (isset($data[0]['TOTAL_LINES'])) {
            $total = $data[0]['TOTAL_LINES'];
        } else if (isset($data['TOTAL_LINES'])) {
            $total = $data['TOTAL_LINES'];
        } else {
            $total = count($data);
        }

        $page = $page ?? $this->getPage();
        $limit = $limit ?? $this->getLimit() ?? $total;

        $pagination = array(
            "total" => (int)$total,
            "page"  => (int)$page,
            "limit" => (int)$limit
        );

        return array(
            'data' => $data,
            'pagination' => $pagination
        );
    }

    /**
     * @param $query
     * @param null $page
     * @param null $limit
     * @param null $order
     * @param null $search
     * @param null $searchIn
     * @return string
     */
    public function getFinalQuery($query, $page = null, $limit = null, $order = null, $search = null, $searchIn = null): string
    {
        $page = $page ?? $this->getPage();
        $limit = $limit ?? $this->getLimit();
        $order = $order ?? $this->getOrder();
        $search = $search ?? $this->getSearch();
        $searchIn = $searchIn ?? $this->getSearchIn();
        $search = $search ? $this->createSearch($search, $searchIn) : '';
        $pagination = $this->createPagination($page, $limit);
        $order = empty($order) ? '' : $this->createOrderBy($order);
        return $this->buildQuery($query, $pagination, $order, $search);
    }

    /**
     * @param EntityManager $entityManager
     * @param string $finalQuery
     * @param array $params
     * @param array $types
     * @param null $page
     * @param null $limit
     * @return array
     */
    public function executeQueryAndBuildResponseData(EntityManager $entityManager, string $finalQuery, $params = [],
                                                     $types = [], $page = null, $limit = null): array
    {
        $data = $entityManager->getConnection()->fetchAll($finalQuery, $params, $types);
        return $this->buildResponseData($data, $page, $limit);
    }

    /**
     * @param EntityManager $entityManager
     * @param string $query
     * @param array $params
     * @param array $types
     * @param null $page
     * @param null $limit
     * @param null $order
     * @return array
     */
    public function createAndExecuteQueryAndBuildResponseData(EntityManager $entityManager, string $query, $params = [],
                                                              $types = [], $page = null, $limit = null, $order = null): array
    {
        $finalQuery = $this->getFinalQuery($query, $page, $limit, $order);
        return $this->executeQueryAndBuildResponseData($entityManager, $finalQuery, $params, $types, $page, $limit);
    }
}
