<?php

namespace Teknisa\Libs\Service;

use Teknisa\Libs\Exception\Login as LoginException;
use Teknisa\Libs\Util\InstanceProvider;
use Teknisa\Libs\Util\Utilities;

class Menu {
    const ALL_ELEMENTS_VISIBLE = 'ALL';
    const ZHU_LOG_PACKAGE = 'zhulog';
    const ZHU_LOG_CONTAINER = 'log';

    protected $menuConfigPath;

    private $menuSession = [];

    public function __construct() {
        $config = json_decode(file_get_contents(__DIR__ . '/../../../../config/config.json'), true);
        $this->menuConfigPath = $config['menuConfigPath'];
    }

    /**
     * @param bool $concatenateGroupMenu
     * @param array $visibleMenus
     * @param array $expiredMenus
     * @param string $urlOrigin
     * @param string $currentModule
     * @param string $currentLanguage
     * @return array
     * @throws \Exception
     */
    public function buildMenu($visibleMenus, $expiredMenus) {
        $modulesConfig  = $this->getJsonMetadata('modules.json', true);
        $menusMetaData  = $this->getMenusMetadata($modulesConfig);
        $menus          = $this->joinMenus($menusMetaData['MENUS']);
        $menus          = $this->mergeMenus($menus, $modulesConfig);
        $this->setMenuSession($menus);
        $this->updateInvisibleAndExpiredMenus($menus, $visibleMenus, $expiredMenus);
        InstanceProvider::getSession()->set('MENU_ROUTES', $this->menuSession);
        $this->menuSession = [];
        return $menus;
    }

    /**
     * @return array
     * @throws \Exception
     */
    private function getMenuConfig()
    {
        $menuConfig = @file_get_contents($this->menuConfigPath);
        if (!$menuConfig) {
            LoginException::menuConfigNotFound();
        }
        return json_decode($menuConfig, true);
    }

    /**
     * @param $menuConfig
     * @return array
     * @throws \Exception
     */
    private function getContainersJson($menuConfig)
    {
        $containers = @file_get_contents($this->updateFilePath($menuConfig['containersFile']));
        if (!$containers) {
            LoginException::containersNotFound();
        }
        return json_decode($containers, true);
    }

    /**
     * @param $modulesFilePath
     * @param bool $isException
     * @throws \Exception
     * @return array
     */
    private function getJsonMetadata($modulesFilePath, $isException = false)
    {
        $modulesConfig = @file_get_contents($this->updateFilePath($modulesFilePath));
        if (!$modulesConfig && $isException) {
            LoginException::modulesConfigNotFound();
        }
        return $modulesConfig ? json_decode($modulesConfig, true) : $modulesConfig;
    }

    /**
     * @param $modulesFilePath
     * @param $currentLanguage
     * @throws \Exception
     * @return array
     */
    private function getTranslationJsonMetadata($modulesFilePath, $currentLanguage)
    {
        $wordsPath    = 'words/' . $currentLanguage . '.json';
        $menuFile     = strpos($modulesFilePath, 'menu.json') ? 'menu.json' : 'aclStruct.json';
        if(!strpos($modulesFilePath, 'modules')) {
            $modulesFilePath = '/mobile/' . $modulesFilePath;
        }
        $languageFile = str_replace($menuFile, $wordsPath, $modulesFilePath);
        $fileContents = @file_get_contents($this->updateFilePath($languageFile));
        return $fileContents ? json_decode($fileContents, true) : $fileContents;
    }

    /**
     * @param $menus
     * @param $devOpsId
     */
    private function addDevOpsIdInMenus(&$menus, $devOpsId) {
        foreach($menus as $key => &$menu) {
            $menu['devOpsId'] = $devOpsId;
            if(isset($menu['items'])) {
                $this->addDevOpsIdInMenus($menu['items'], $devOpsId);
            }
        }
    }

    /**
     * @param $menuConfig
     * @param $modulesConfig
     * @param $currentLanguage
     * @return array
     */
    private function getMenusMetadata($modulesConfig)
    {
        foreach ($modulesConfig as $moduleName => $modulePath) {
            if ($moduleName === 'product') {
                $menu = $this->getJsonMetadata('menu.json');
            } else {
                $menu = $this->getJsonMetadata("./modules/{$moduleName}/menu.json");
            }
            if ($menu) {
                $this->addDevOpsIdInMenus($menu, $modulesConfig[$moduleName]['devOpsId']);
                $menus[] = $menu;
            }
        }

        return array(
            'MENUS' => $menus
        );
    }

    /**
     * @param $menus
     * @return array
     */
    private function joinMenus($menus)
    {
        $menusAux = array();
        foreach ($menus as $menuGrouped) {
            $menusAux = array_merge($menusAux, $menuGrouped);
        }
        return $menusAux;
    }

    /**
     * @param $menus
     * @param $filterCache
     * @param $modulesConfig
     * @param $modulesTranslations
     * @return array
     */
    private function filterMenus($menus, $filterCache, $modulesConfig) {
        $menusAux = [];
        foreach($menus as $menu) {
            $show = true;
            $menuName = isset($menu['name']) ? $menu['name'] : $menu['label'];
            if(isset($filterCache[$menuName])) {
                $show = false;
                $cachedMenu = $filterCache[$menuName];
                if(isset($cachedMenu['items'])) {
                    $cachedMenu['items'] = $this->overrideMenus($cachedMenu, $menu);
                    foreach($menusAux as &$menuAux) {
                        $menuAuxName = isset($menuAux['name']) ? $menuAux['name'] : $menuAux['label'];
                        $cachedMenuName = isset($cachedMenu['name']) ? $cachedMenu['name'] : $cachedMenu['label'];
                        if($menuAuxName == $cachedMenuName) {
                            $menuAux['items'] = array_merge($menuAux['items'], $cachedMenu['items']);
                        }
                    }
                }
            } else {
                $cachedMenu = $filterCache[$menuName] = $menu;
            }

            if(isset($cachedMenu['items'])) {
                foreach($cachedMenu['items'] as $childrenMenu) {
                    if(isset($childrenMenu['items'])) {
                        $cache = [];
                        $childrenMenu['items'] = $this->filterMenus($childrenMenu['items'], $cache, $modulesConfig);
                    }
                }
            }

            if($show) {
                $menusAux[] = $menu;
            }
        }

        return $menusAux;
    }

    private function updateMenuURL($menu, $urlOrigin, $modulesConfig, $currentModule) {
        if(isset($menu['windowName'])) {
            $hash   = strpos($menu['windowName'], '#') ? '#' : '%23';
            $window = explode($hash, $menu['windowName']);
            if(count($window) == 1) {
                $module = $currentModule;
            } else {
                $module = explode('_', $window[0])[0];
            }

            if($module != $currentModule && isset($modulesConfig[$module]['baseUrl'])) {
                $menu['url'] = $urlOrigin . '/' . $modulesConfig[$module]['baseUrl'] . "/#/" . $menu['windowName'];
                $menu['msContainerName'] = $menu['windowName'];
                $menu['events'] = array(
                    array(
                        'name' => 'MenuOnClick',
                        'code' => 'LibUtilities.openMicroServiceUrl(args)'
                    )
                );
                unset($menu['windowName']);
            }
        }

        return $menu;
    }

    /**
     * @param $uncheckedMenus
     * @param $urlOrigin
     * @param $modulesConfig
     * @param $currentModule
     * @param $modulesTranslations
     * @return array
     */
    private function mergeMenus($uncheckedMenus, $modulesConfig) {
        $filterCache  = [];
        $checkedMenus = $this->filterMenus($uncheckedMenus, $filterCache, $modulesConfig);
        foreach($checkedMenus as &$checkedMenu) {
            if(isset($checkedMenu['items'])) {
                $checkedMenu['items'] = $this->mergeMenus($checkedMenu['items'], $modulesConfig);
            }
        }

        return $checkedMenus;
    }

    /**
     * @param $cachedMenu
     * @param $menu
     * @return array
     */
    private function overrideMenus($cachedMenu, $menu)
    {
        $cachedMenu['items'] = array_merge($cachedMenu['items'], $menu['items']);
        $menusToOverride = array_filter($cachedMenu['items'], function ($menu) {
            return isset($menu['override']);
        });
        $cachedMenusAux = [];
        foreach ($cachedMenu['items'] as $cached) {
            $shouldFilter = false;
            foreach ($menusToOverride as &$menuToOverride) {
                $overrideMenuId = isset($menuToOverride['id']) ? $menuToOverride['id'] : null;
                $cachedMenuId = isset($cached['id']) ? $cached['id'] : false;
                $shouldFilter = $cachedMenuId        !== $overrideMenuId &&
                    ((isset($cached['windowName']) && $cached['windowName'] == $menuToOverride['override']) ||
                        $cached['name'] == $menuToOverride['override']);
                if ($shouldFilter && isset($cached['items']) && isset($menuToOverride['items'])) {
                    $menuToOverride['items'] = array_merge($menuToOverride['items'], $cached['items']);
                }
                if ($shouldFilter) {
                    break;
                }
            }
            if (!$shouldFilter) {
                $cachedMenusAux[] = $cached;
            }
        }
        return $cachedMenusAux;
    }

    /**
     * @param $menus
     * @param $visibleMenus
     * @param $expiredMenus
     */
    private function updateInvisibleAndExpiredMenus(&$menus, $visibleMenus, $expiredMenus) {
        $allMenus = !empty($visibleMenus) && $visibleMenus[0] == self::ALL_ELEMENTS_VISIBLE;
        foreach($menus as $key => &$menu) {
            if ($this->isExpiredMenu($expiredMenus, $menu, $allMenus)) {
                $menu['icon']     = 'mdi-alert-circle';

                if(isset($menu['items'])) {
                    $this->updateInvisibleAndExpiredMenus($menu['items'], $visibleMenus, $expiredMenus);
                } else {
                    unset($menu['route']);
                    $menu['cssStyle'] = 'opacity: 0.4;';
                    $menu['events'] = array(
                        'click' => '{{AppController.showMessageExpiredMenu}}'
                    );
                }
            } else if (!$allMenus && $this->isInvisibleMenu($visibleMenus, $menu, $allMenus) && $this->isInvisibleMenuChild($visibleMenus, $menu, $allMenus)) {
                unset($menus[$key]);
            } else {
                if(isset($menu['items'])) {
                    $this->updateInvisibleAndExpiredMenus($menu['items'], $visibleMenus, $expiredMenus);
                }

                $menu['isVisible'] = true;

                if (isset($menu['route'])) {
                    $this->menuSession[$menu['route']] = true;

                    if (isset($menu['dependency'])) {
                        if (is_array($menu['dependency'])) {
                            foreach ($menu['dependency'] as $route) {
                                $this->menuSession[$route] = true;
                            }
                        } else {
                            $this->menuSession[$menu['dependency']] = true;
                        }
                    }
                }
            }
        }
        $menus = array_values($menus);
    }

    private function isInvisibleMenuChild($visibleMenus, &$menu, $allMenus)
    {
        if (isset($menu['items'])) {
            foreach ($menu['items'] as $item) {
                if (!$this->isInvisibleMenu($visibleMenus, $item, $allMenus)) {
                    return false;
                } else {
                    if (isset($item['items'])) {
                        return $this->isInvisibleMenuChild($visibleMenus, $item, $allMenus);
                    }
                }
            }
        }
        return true;
    }

    /**
     * @param $menuConfig
     * @param $aclStruct
     * @return array
     */
    private function concatenateGroupMenu($menuConfig, $aclStruct, $modulesConfig)
    {
        $containers = $this->getContainersJson($menuConfig);
        $menus = [];
        foreach ($containers as $project) {
            if (isset($project['groupMenu'])) {
                foreach ($project['groupMenu'] as &$groupMenu) {
                    $groupMenu['devOpsId'] = $modulesConfig['product']['devOpsId'];
                    foreach ($aclStruct as $struct) {
                        if (isset($struct['parent']) && $struct['parent'] == $groupMenu['name']) {
                            $groupMenu['menus'][] = $struct;
                        }
                    }
                }
                $menus = $project['groupMenu'];
            }
        }
        return $menus;
    }

    /**
     * @param $visibleMenus
     * @param $menu
     * @param $allMenus
     * @return bool
     */
    private function isInvisibleMenu($visibleMenus, $menu, $allMenus)
    {
        $userAclLabels = InstanceProvider::getSession()->get('USER_ACL_LABELS') ?? [];
        $isAdm = !is_dir(Utilities::getProductBasePath() . '/modules/acl') || InstanceProvider::getSession()->get('ACL_ADMINISTRATOR');

        return (!$allMenus &&
            (
                (isset($menu['isVisible']) && $menu['isVisible'] == false) ||
                (!isset($menu['id'])) ||
                ($visibleMenus !== null && !in_array($menu['id'], $visibleMenus)) ||
                (!$isAdm && !in_array($menu['id'], $userAclLabels))
            )
        );
    }

    /**
     * @param $expiredMenus
     * @param $menu
     * @param $allMenus
     * @return bool
     */
    private function isExpiredMenu($expiredMenus, $menu, $allMenus)
    {
        return (!$allMenus &&
            (
                $expiredMenus &&
                (isset($menu['isVisible']) && $menu['isVisible']) &&
                (isset($menu['id']) && in_array($menu['id'], $expiredMenus))
            )
        );
    }

    private function updateFilePath($path) {
        return Utilities::getProductBasePath() . $path;
    }

    /**
     * @param $menuConfig
     * @param $modulePath
     * @param $moduleName
     * @return array
     */
    private function getRouteTranslate($menuConfig, $modulePath, $moduleName)
    {
        $modulePathArr = explode('/', $modulePath);
        if ($moduleName == Utilities::PRODUCT_MENU_NAME) {
            $modulePathArr = explode('/', $menuConfig['parentAclStructFile']);
        }
        array_pop($modulePathArr);
        $routeTranslateFile = join('/', $modulePathArr) . '/routeTranslate.json';
        return $this->getJsonMetadata($routeTranslateFile);
    }

    private function generateRoutesTranslateDataSource($routesTranslate) {
        array_walk($routesTranslate, function(&$routeLabel, $route) {
            $routeLabel = array(
                'ROUTE' => $route,
                'ROUTE_LABEL' => $routeLabel
            );
        });

        return array_values($routesTranslate);
    }

    private function setMenuSession($menus)
    {
        foreach ($menus as $item) {
            if (isset($item['route'])) {
                $this->menuSession[$item['route']] = false;
                if (isset($item['dependency'])) {
                    if (is_array($item['dependency'])) {
                        foreach ($item['dependency'] as $route) {
                            $this->menuSession[$route] = false;
                        }
                    } else {
                        $this->menuSession[$item['dependency']] = false;
                    }
                }
            }

            if (!empty($item['items'])) {
                $this->setMenuSession($item['items']);
            }
        }
    }
}
