<?php

namespace Teknisa\Libs\Controller;

use Teknisa\Libs\Exception\CustomException;
use Teknisa\Libs\Util\Utilities;
use Teknisa\Libs\Util\InstanceProvider;
use Zeedhi\Framework\DTO\Request;
use Zeedhi\Framework\DTO\Response;
use Zeedhi\Framework\DataSource\DataSet;
use Teknisa\Libs\Service\Login as LoginService;
use Teknisa\Libs\Service\Api as ApiService;
use Teknisa\Libs\Util\Utilities as LibUtilities;
use Teknisa\Libs\Traits\TResponse;

class Login {
    use TResponse;

    /** @var LoginService $loginService */
    private $loginService;
    /** @var ApiService $apiService */
    private $apiService;

    /**
     * @param LoginService $loginService
     * @param ApiService $apiService
     */
    public function __construct(LoginService $loginService, ApiService $apiService) {
        $this->loginService = $loginService;
        $this->apiService   = $apiService;
    }

    /**
     * @param Request $request
     * @param Response $response
     * @throws \Exception
     */
    public function startSession(Request $request, Response $response) {
        try {
            $params = $request->getParameters();
            $loginUser  = $params['USER'];
            $loginToken = $params['LOGIN_TOKEN'];
            $loginLang  = $params['LOGIN_LANGUAGE'];
            $loginSep   = empty($params['LOGIN_SEPARATOR']) ? null : $params['LOGIN_SEPARATOR'];

            $modulesVersionError = null;
            try {
                $this->apiService->getModulesVersion();
            } catch(CustomException $e) {
                $modulesVersionError = array(
                    'MESSAGE_CODE'   => $e->getMessage(),
                    'MESSAGE_PARAMS' => $e->getParams()
                );
            }

            $filter = array(
                'TOKEN' => $loginToken
            );
            $licenseInfo = false;
            $redirectionData = InstanceProvider::getMongoDB()->find('LoginRedirection', $filter);
            if(!empty($redirectionData)) {
                $loginUser = $redirectionData[0]['USER'];
                if(!empty($redirectionData[0]['LICENSE_INFO'])) {
                    $licenseInfo = $redirectionData[0]['LICENSE_INFO'];
                }
                InstanceProvider::getMongoDB()->remove('LoginRedirection', $filter);
            }
            

            $loginUserInfo = @json_decode(LibUtilities::zeedhiDecrypt($loginUser), true);
            if(empty($loginUserInfo)) {
                $loginUserInfo = InstanceProvider::getSession()->get('LOGIN_USER_INFO');
            } else {
                unset($loginUserInfo['CURRENT_SECONDS']);
            }
            if(isset($loginUserInfo['USER_DATA']) && !empty($loginUserInfo['USER_DATA'])) {
                $userData = $loginUserInfo['USER_DATA'];
            } else {
                $userData = $this->apiService->getUserData($loginUserInfo['USER_EMAIL'], null);
            }

            $noticeForOperatorData = InstanceProvider::getMongoDB()->find('LoginNoticeForOperator', $filter);
            $notices = [];
            $timeOutNotices = [];
            if(!empty($noticeForOperatorData)) {
                $notices = $noticeForOperatorData[0]['NOTICES'];
                $timeOutNotices = $noticeForOperatorData[0]['TIMEOUTNOTICES'];
            }

            $userData = $this->loginService->restoreSession($userData, $loginUserInfo, $modulesVersionError, $loginToken, $loginLang, $loginSep, $notices, $timeOutNotices);
            $userData['LICENSE_INFO'] = $licenseInfo;
            $this->sendResponse($userData, 200);
        } catch(CustomException $e) {
            $this->sendResponse(Utilities::getErrorObjectFromException($e), 500);
        }
    }

    /**
     * @param Request $request
     * @param Response $response
     * @throws \Exception
     */
    public function logout(Request $request, Response $response) {
        try {
            $params = $request->getParameters();
            $project    = $params['PRODUCT_ID'] || null;
            $hash       = $params['HASH'] || null;
            $this->loginService->logout($project, $hash);
        } catch(\Exception $e) {
            $this->sendResponse(['error' => $e->getMessage(), 'code' => $e->getCode()], 500);
        }
    }

    /**
     * @param Request $request
     * @param Response $response
     * @throws \Exception
     */
    public function validateExpiredSession(Request $request, Response $response) {
        try {
            $timeout     = $request->getParameter('TIMEOUT');
            $sessionInfo = $this->loginService->validateExpiredSession($timeout);
            $this->sendResponse($sessionInfo, 200);
        } catch(\Exception $e) {
            $response->setError(new Response\Error($e->getMessage(), Response\Message::TYPE_ERROR));
        }
    }

    /**
     * @param Request $request
     * @param Response $response
     * @throws \Exception
     */
    public function updateSession(Request $request, Response $response) {
        try {
            $params             = $request->getParameters();
            $user               = $params['USER'];
            $accessProfile      = $params['ACCESS_PROFILE'];
            $otherSessionValues = $params['OTHER_SESSION_VALUES'];
            $loginUser          = $params['LOGIN_USER'];
            $userSession        = $this->loginService->updateSession($user, $accessProfile, $otherSessionValues, $loginUser);
            $this->sendResponse($userSession, 200);
        } catch(\Exception $e) {
            $this->sendResponse(['error' => $e->getMessage(), 'code' => $e->getCode()], 500);
        }
    }

    /**
     * @param Request $request
     * @param Response $response
     * @throws \Exception
     */
    public function saveCollectedData(Request $request, Response $response){
        try {
            $params         = $request->getParameters();
            $collectionIds  = $params['COLLECTION_IDS'];
            $operationId    = $params['OPERATION_ID'];
            $productId      = $params['PRODUCT_ID'];
            $operatorCode   = $params['OPERATOR_CODE'];
            $operatorName   = $params['OPERATOR_NAME'];
            $organizationId = $params['ORGANIZATION_ID'];
            Utilities::saveCollectedData($collectionIds, $operationId, $productId, $operatorCode, $operatorName, $organizationId);
        } catch(\Exception $e) {
            $this->sendResponse(['error' => $e->getMessage(), 'code' => $e->getCode()], 500);
        }
    }

    /**
     * @param Request $request
     * @param Response $response
     * @throws \Exception
     */
    public function saveProductsPermission(Request $request, Response $response){
        try {
            $row         = $request->getParameters();
            $licenseInfo = $row['LICENSE_INFO'];
            $licenseInfo = json_decode(LibUtilities::zeedhiDecrypt($licenseInfo), true);
            $this->loginService->saveProductsPermission($licenseInfo);
        } catch(\Exception $e) {
            $this->sendResponse(['error' => $e->getMessage(), 'code' => $e->getCode()], 500);
        }
    }

    /**
     * @param Request $request
     * @param Response $response
     * @throws \Exception
     */
    public function saveVariableMeasure(Request $request, Response $response){
        try {
            $row         = $request->getParameters();
            $licenseInfo = $row['LICENSE_INFO'];
            $licenseInfo = json_decode(LibUtilities::zeedhiDecrypt($licenseInfo), true);
            $variableMeasure = $generalCollection = array();

            if(!empty($licenseInfo['VARIABLE_MEASURE'])) {
                $variableMeasure = $this->loginService->getVariableMeasureRegisters($licenseInfo);
            }

            if(!empty($licenseInfo['GENERAL_COLLECTION'])) {
                $generalCollection = $this->loginService->getGeneralCollectionRegisters($licenseInfo);
            }

            if(!empty($variableMeasure) || !empty($generalCollection)) {
                $this->loginService->saveVariableMeasure($variableMeasure, $generalCollection, $licenseInfo);
            }
        } catch(\Exception $e) {
            $this->sendResponse(['error' => $e->getMessage(), 'code' => $e->getCode()], 500);
        }
    }

    /**
     * @param Request $request
     * @param Response $response
     * @throws \Exception
     */
    public function setLogAccess(Request $request, Response $response){
        try {
            $this->loginService->setLogAccess();
        } catch(\Exception $e) {
            $this->sendResponse(['error' => $e->getMessage(), 'code' => $e->getCode()], 500);
        }
    }

}
