<?php

namespace Teknisa\Libs\Service;

use Teknisa\Libs\Util\Environment;
use Teknisa\Libs\Util\InstanceProvider;
use Teknisa\Libs\Util\Utilities;

class Metadata
{
    const DS = DIRECTORY_SEPARATOR;
    const JSON_NAME_NOTFOUND = 'notfound';
    const JSON_NAME_UNAUTHORIZED = 'unauthorized';
    const DEFAULT_TEMPLATE_PROPS = [
        "isVisible" => true,
        "addButton" => true,
        "filterButton" => true,
        "columnFilterButton" => true,
        "columnsButton" => true,
        "showLayoutOptions" => true,
        "showReload" => true,
        "fillHeight" => true,
        "doubleClickEdit" => true,
        "lazyLoad" => false,
        "deleteButton" => "selection",
        "height" => "100%",
        "maxHeight" => "18.75rem",
        "addClick" => "",
        "onCreated" => "",
        "onMounted" => "",
        "submit" => "",
        "beforeDelete" => "",
        "cellClick" => "",
        "rowClick" => "",
        "beforeCancel" => "",
        "beforeSave" => "",
        "onBeforeDestroy" => "",
        "aclLabel" => "",
        "uniqueKey"=> "id",
        "type" => "tek-rest",
        "route" => "",
        "limit" => 50,
        "path" => "",
        "value" => "",
        "data" => [],
        "order" => [],
        "searchIn" => [],
        "filter" => [],
        "find" => [],
        "rightSlot" => [],
        "leftSlot" => []     
    ];
    
    /**
     * @var Environment
     */
    private $environment;

    public function __construct(Environment $environment)
    {
        $this->environment = $environment;
    }

    public function findMetadata($jsonPath, $override = null, $routePath = null)
    {
        $isJsonFragment = !$routePath;
        if ($this->isJsonTemplate($jsonPath)) {
            $jsonMetadata = $this->getJsonMetadataByTemplate($jsonPath);
        } else if (Utilities::isModule()) {
            $jsonMetadata = $this->getJsonMetadataByModule($jsonPath);
        } else {
            $jsonMetadata = $this->getJsonMetadataByProduct($jsonPath);
        }

        if (!$jsonMetadata) {
            return $this->getJsonMetadataByProduct(self::JSON_NAME_NOTFOUND);
        }

        if (!$isJsonFragment && $this->unauthorizedAccessToJsonMetadata($routePath)) {
            return $this->getJsonMetadataByProduct(self::JSON_NAME_UNAUTHORIZED);
        }

        $notAuthorized = $this->filterMetadataByAcl($jsonMetadata, $override, $jsonPath);

        if ($notAuthorized && $isJsonFragment) {
            return [];
        }

        if ($notAuthorized && !$isJsonFragment) {
            return $this->getJsonMetadataByProduct(self::JSON_NAME_UNAUTHORIZED);
        }

        $this->removeEmptyEvents($jsonMetadata);
        return $jsonMetadata;
    }

    private function removeEmptyEvents(&$metadata) {        
        foreach ($metadata as $key => &$data) {
            if(is_array($data)) {
                if($key == 'events' && empty($data)) {
                    unset($metadata[$key]);
                } else {
                    $this->removeEmptyEvents($data);
                }
            }
        }
    }

    private function isJsonTemplate($jsonPath) {
        return strpos($jsonPath, 'templates/') !== false;
    }

    /**
     * @param $metadata
     * @return bool
     */
    private function filterMetadataByAcl(&$metadata, $override, $jsonPath): bool
    {
        foreach ($metadata as $key => &$data) {                            
            $this->overrideTemplateKeyInData($data, $key, $override, $metadata, $jsonPath);
            if (is_array($data)) {
                $notAuthorized = $this->filterMetadataByAcl($data, $override, $jsonPath);
                if ($notAuthorized) {
                    unset($metadata[$key]);
                    $metadata = array_values($metadata);                    
                }
            } else {
                /* Examples:
                    1) "addButton":    "ACL_LABEL|LABEL1,LABEL2"          -> return boolean: true or false
                    2) "deleteButton": "ACL_LABEL|LABEL1|selection,none"  -> return string: selection or none
                    3) "intValue":     "ACL_LABEL|LABEL1,LABEL2|1,2|int"  -> return int: 1 or 2
                    4) "floatValue":   "ACL_LABEL|LABEL1|1.01,2.99|float" -> return float: 1.01 or 2.99
                    5) "boolValue":    "ACL_LABEL|LABEL1|false,true|bool" -> return boolean: false or true
                */
                if(is_string($data) && strpos($data, 'ACL_LABEL') !== false) {
                    $labelInfo = explode('|', $data);
                    $labels = $labelInfo[1];
                    $values = isset($labelInfo[2]) ? explode(',', $labelInfo[2]) : array(true, false);
                    $data = $this->checkLabelInSession($labels) ? $values[0] : $values[1];
                    $valueType = isset($labelInfo[3]) ? $labelInfo[3] : null;
                    if($valueType == 'int') {
                        $data = intval($data);
                    } else if($valueType == 'float') {
                        $data = floatval($data);
                    } else if($valueType == 'bool') {
                        $data = $data == 'true' ? true : false;
                    }
                } else if ($key === 'aclLabel' && !$this->checkLabelInSession($data)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * @param $aclLabel
     * @return bool
     */
    public function checkLabelInSession($aclLabel): bool
    {
        $isAdm = InstanceProvider::getSession()->get('ACL_ADMINISTRATOR');
        $userAclLabels = InstanceProvider::getSession()->get('USER_ACL_LABELS');
        if($isAdm || is_null($userAclLabels) || empty($aclLabel)) {
            return true;
        }

        $labels = explode(',', $aclLabel);
        foreach($labels as $label) {
            if(in_array($label, $userAclLabels)) {
                return true;
            }
        }
        return false;
    }

    private function overrideTemplateKeyInData(&$data, $key, $override, &$metadata, $jsonPath): void {
        if (!is_null($override) && !empty($override)) {
            if ((!is_array($data) || $this->isFormReport($jsonPath)) && !empty($data)) {
                foreach ($override as $keyOverride => $valueOverride) {
                    if($this->isFormReportSlots($jsonPath, $key, $keyOverride)) {
                        $data = $keyOverride === 'RIGHT_SLOT' ? array_merge($valueOverride, $data) : array_merge($data, $valueOverride);
                        break;
                    } else if($this->isFormReportContent($jsonPath, $data, $keyOverride)) {
                        $data['children'] = array_merge($data['children'], $valueOverride);
                        break;
                    } else if ($this->checkValueToOverride($data, $keyOverride)) {
                        if (is_bool($valueOverride) || is_array($valueOverride)) {
                            $data = $valueOverride;
                            break;
                        }
                        $data = str_replace('<<'.$keyOverride.'>>', $valueOverride, $data);
                        break;
                    }
                }
                if (!is_array($data) && strpos($data, '<<') !== false && isset(self::DEFAULT_TEMPLATE_PROPS[$key])) {                                           
                    $data = self::DEFAULT_TEMPLATE_PROPS[$key];
                }
                if(empty($data) && $data !== false) {
                    unset($metadata[$key]);
                }
            }
        }
    }

    private function isFormReport($jsonPath) {
        return $jsonPath === 'templates/form-report';
    }

    private function isFormReportSlots($jsonPath, $key, $keyOverride) {
        $isRigthSlot = $jsonPath === 'templates/form-report' && $key === 'rightSlot' && $keyOverride === 'RIGHT_SLOT';
        $isLeftSlot  = $jsonPath === 'templates/form-report' && $key === 'leftSlot'  && $keyOverride === 'LEFT_SLOT';
        return $isRigthSlot || $isLeftSlot;
    }

    private function isFormReportContent($jsonPath, &$data, $keyOverride) {
        return $jsonPath === 'templates/form-report' && is_array($data) && $keyOverride === 'FORM' && !empty($data['isFormReport']);
    }

    private function checkValueToOverride($data, $keyOverride) {
        if(is_array($data)) {
            return false;
        }
        if($keyOverride === 'NAME' || $keyOverride === 'MODAL_NAME') {
            return strpos($data, '<<' .$keyOverride .'>>') !== false;
        }
        return $data === '<<'.$keyOverride.'>>';
    }

    /**
     * @param $name
     * @return string
     */
    private function getJsonMetadataPathByProduct($name): string
    {
        return Utilities::getProductBasePath() . self::DS . Utilities::getJsonMetadataPathByProductParameter() . self::DS . $name . '.json';
    }

    /**
     * @param $name
     * @return string
     * Mudar método posteriormente
     */
    private function getJsonMetadataPathByModule($name): string
    {
        return Utilities::getProductBasePath() . self::DS . Utilities::getJsonMetadataPathByModuleParameter() . self::DS . $name . '.json';
    }

    private function getJsonMetadataPathByTemplate($name): string
    {
        return __DIR__ . self::DS . '..' . self::DS . $name . '.json';
    }


    /**
     * @param $jsonPath
     * @return mixed
     */
    private function getJsonMetadataByProduct($jsonPath)
    {
        $jsonMetadataPath = $this->getJsonMetadataPathByProduct($jsonPath);
        return @json_decode(file_get_contents($jsonMetadataPath), true);
    }

    private function getJsonMetadataByModule($jsonPath)
    { 
        $jsonMetadataPath = $this->getJsonMetadataPathByModule($jsonPath);
        return @json_decode(file_get_contents($jsonMetadataPath), true);
    }

    private function getJsonMetadataByTemplate($jsonPath)
    {
        $jsonMetadataPath = $this->getJsonMetadataPathByTemplate($jsonPath);
        return @json_decode(file_get_contents($jsonMetadataPath), true);
    }

    /**
     * @param $routePath
     * @return bool
     */
    private function unauthorizedAccessToJsonMetadata($routePath): bool
    {
        $menuRoutes = InstanceProvider::getSession()->get('MENU_ROUTES');
        return isset($menuRoutes[$routePath]) && !$menuRoutes[$routePath];
    }

    /**
     * @param $routePath
     * @return bool
     */
    private function hasIndexInMenuRoutesSession($routePath): bool
    {
        $menuRoutes = InstanceProvider::getSession()->get('MENU_ROUTES');
        return isset($menuRoutes[$routePath]);
    }
}