<?php

namespace Zeedhi\DataExporter\Service\ExportStrategy;

use Zeedhi\DataExporter\Service\DataExporter;
use Zeedhi\DataExporter\Utility\UniqueFileNameProvider;
use Zeedhi\DataExporter\Service\ExportStrategy;

/**
 * Manages the writing on spreadsheet files.
 */
class SpreadSheet implements ExportStrategy {

    /** @const string Delimit a sheet to write on the xls. */
    const SHEET_NAME = 'sheet1';

    /** @var \XLSXWriter Writes values on a xls file. */
    private $xlsxWriter;

    /** @var UniqueFileNameProvider Provides name and path to the report.*/
    private $fileNameProvider;

    /** @var Array Describes the structure of the report. */
    private $metadata;

    /**
     * Constructor.
     *
     * @param UniqueFileNameProvider $fileNameProvider
     */
    public function __construct(UniqueFileNameProvider $fileNameProvider)
    {
        $this->fileNameProvider = $fileNameProvider;
    }
    
    protected $momentToPhpDateDictionary = [
        'M'     => 'n',
        'Mo'    => '',
        'MM'    => 'm',
        'MMM'   => 'M',
        'MMMM'  => 'F',
        'Q'     => '',
        'Qo'    => '',
        'D'     => 'j',
        'Do'    => 'jS',
        'DD'    => 'd',
        'DDD'   => 'z',
        'DDDo'  => '',
        'DDDD'  => '',
        'd'     => 'w',
        'do'    => '',
        'dd'    => 'd',
        'ddd'   => '',
        'dddd'  => '' ,
        'e'     => 'w',
        'E'     => '',
        'w'     => '',
        'wo'    => '',
        'ww'    => '',
        'W'     => '',
        'Wo'    => '',
        'WW'    => '',
        'YY'    => 'y',
        'YYYY'  => 'Y',
        'Y'     => 'Y',
        'yyyy'  => 'Y',
        'yy'    => 'y',
        'gg'    => '',
        'gggg'  => '',
        'GG'    => '',
        'GGGG'  => '',
        'A'     => 'A',
        'a'     => 'a',
        'H'     => 'G',
        'HH'    => 'H',
        'h'     => 'g',
        'hh'    => 'h',
        'k'     => '',
        'kk'    => '',
        'm'     => '',
        'mm'    => 'm',
        's'     => '',
        'ss'    => 's',
        'S'     => '',
        'z'     => 'T',
        'zz'    => 'T',
        'Z'     => 'P',
        'ZZ'    => 'O',
        'X'     => 'U',
        'x'     => ''
    ];

    /**
     * Get a XLS writer.
     *
     * Returns a new XLSWriter.
     *
     * @return XLSXWriter   A XLS writer.
     */
    public function getNewXLSWriter() {
        return new \XLSXWriter();
    }

    private function configMasks(){
        return [
            "date" => [
                "format" => "dd/mm/yyyy"
            ],
            "datetime" => [
                "format" => "dd/mm/yyyy"
            ]
        ];
    }
    
    /**
     * Write the header with the types of each column.
     * 
     * @param array $metadata   Metadata for the file creation.
     */
    public function writeHeader(array $metadata){
        $columns = $this->prepareRow($metadata['columns']);
        if(isset($metadata['xlsDefaultType'])){
            foreach($columns as $column){
                if(isset($column['description'])){
                    $preparedHeader[$column['description']] = key_exists('xlsType', $column) ? 
                        $column['xlsType'] : $metadata['xlsDefaultType'];
                }
            }
            $columns = $preparedHeader;
            $this->xlsxWriter->writeSheetHeader(self::SHEET_NAME, $columns);
        }
        else{
            foreach($columns as $column){
                $preparedHeader[] = isset($column['description']) ? $column['description'] : "";
            }
            $this->xlsxWriter->writeSheetRow(self::SHEET_NAME, array_values($preparedHeader));
        }
    }
    /**
     * Start writing the file.
     *
     * Sort the columns of the metadata by its 'sequence', maps by it's description and calls xlsWritter.
     *
     * @param  array  $metadata   Metadata for the file creation.
     */
    public function start(array $metadata)
    {
        $this->metadata = $metadata;
        $this->xlsxWriter = $this->getNewXLSWriter();
        $this->writeHeader($metadata);
    }
    
    /**
     * Prepare the row to be written.
     *
     * Prepare the row by ordering it's columns based on the metadata.
     *
     * @param  array $row   Row to be prepared.
     *
     * @return array Prepared row.
     */
    protected function prepareRow(array &$row, $masks = []) {
        $preparedRow = array();
        foreach ($this->metadata['columns'] as $name => $column) {
            $row[$name] = isset($row[$name]) ? $row[$name] : "";
            
            if(!empty($row[$name]) && !is_array($row[$name])) {
                $row[$name] = $this->formatValue($row[$name], $column, $masks);
            }
            
            $preparedRow[$column['sequence']] = $row[$name];
        }
        ksort($preparedRow);
        return $preparedRow;
    }
    
    private function translateDateFormat($format){
        return strtr($format, $this->momentToPhpDateDictionary);
    }
    
    protected function formatValue($value, $column, $masks){
        if (!isset($column['format']) || !isset($column['format']['type']) || empty($masks)) return $value;
        if($column['format']['type'] == 'date' || $column['format']['type'] == 'datetime') {
            $format = $this->translateDateFormat($masks[$column['format']['type']]['format']);
            if(strpos($format, ' ') === false) {
                $value = explode(' ', $value)[0];
            }
            $d = \DateTime::createFromFormat($format, $value);
            return $d->format($format);
        }
        return $value;
    }
    
    /**
     * Write the given rows on the xls.
     *
     * Pass through the given rows and write on the xls writter.
     *
     * @param array $rows   Rows to be written on the xls.
     */
    public function writeRows(array $rows) {
        $masks = $this->configMasks();
        foreach($rows as &$row) {
            $writeRow = $this->prepareRow($row, $masks);
            $this->xlsxWriter->writeSheetRow(self::SHEET_NAME, array_values($writeRow));
        }
    }

    /**
     * Finishes the file creation.
     *
     * Generates the file name, get the file full path and write the xls writter on the file.
     *
     * @return string Name of the created file.
     */
    public function finish()
    {
        $fileName = $this->fileNameProvider->generateFileName($this->formatName(), $this->metadata['reportName']);
        $filePath = $this->fileNameProvider->getFullFilePath($fileName);
        $this->xlsxWriter->writeToFile($filePath);
        return $filePath;
    }

    /**
     * Get the spreadsheet file format.
     *
     * @return string The format of the spreadsheet file.
     */
    public function formatName()
    {
        return DataExporter::EXPORT_FORMAT_SPREADSHEET;
    }
}