<?php
namespace Zeedhi\DataExporter\Service;

use Zeedhi\DataExporter\Service\ExportStrategy\ReportStrategy;
use Zeedhi\DataExporter\Service\FileTrem;
use Zeedhi\DataExporter\Exception\Exception;
use Zeedhi\DataExporter\Utility\UniqueFileNameProvider;
use Zeedhi\Report\ReportBuilder;
use Zeedhi\Report\LogoConfig;

class ReportStrategyTest extends \PHPUnit_Framework_TestCase {

    /** @var ReportStrategy */
    private $exportStrategy;
    /** @var array */
    private $metaData;
    /** @var \Zeedhi\Report\ReportBuilder|\PHPUnit_Framework_MockObject_MockObject */
    private $reportBuilder;
    /** @var SplFileObject|\PHPUnit_Framework_MockObject_MockObject */
    private $splFileObject;
    /** @var UniqueFileNameProvider|\PHPUnit_Framework_MockObject_MockObject */
    private $fileNameProvider;
    /** @var \Zeedhi\Report\LogoConfig|\PHPUnit_Framework_MockObject_MockObject */
    private $logoConfig;

    protected function setUp() {
        $this->reportBuilder = $this->getMockBuilder(ReportBuilder::class)
            ->disableOriginalConstructor()
            ->setMethods(array("buildReport"))
            ->getMock();

        $this->splFileObject = $this->getMockBuilder(\SplFileObject::class)
            ->setMethods(array("flock", "fread", "getSize", "fwrite", "rewind", "ftruncate", "fflush", "fseek", "getFilename"))
            ->setConstructorArgs(array('tests/reports/reportTest'))
            ->getMock();

        $this->fileNameProvider = $this->getMockBuilder(UniqueFileNameProvider::class)
            ->disableOriginalConstructor()
            ->setMethods(array("generateFileName", "getFullFilePath"))
            ->getMock();

        $this->logoConfig = $this->getMockBuilder(LogoConfig::class)
            ->disableOriginalConstructor()
            ->setMethods(array("setClientLogo", "setEnterpriseLogo", "setProductLogo"))
            ->getMock();

        $this->metaData = array(
            "title" => "Test Report",
            "columns" => array(
                "value" => array(
                    "size"        => "30%",
                    "align"       => "left",
                    "sequence"    => 2,
                    "description" => "Value"
                ),
                "name" => array(
                    "size"        => "70%",
                    "align"       => "left",
                    "sequence"    => 1,
                    "description" => "Name"
                )
            ),
            "reportName" => "report"
        );

        $this->exportStrategy = $this->getMockBuilder(ReportStrategy::class)
            ->setConstructorArgs(array($this->fileNameProvider))
            ->setMethods(array("getNewReportBuilder", "getNewLogoConfig", "getNewSplFileObject", "writeRows", "finish"))
            ->getMock();

        $this->exportStrategy->method("getNewReportBuilder")->willReturn($this->reportBuilder);
        $this->exportStrategy->method("getNewLogoConfig")->willReturn($this->logoConfig);
        $this->exportStrategy->method("getNewSplFileObject")->willReturn($this->splFileObject);
        $this->exportStrategy->method("finish")->willReturn('fileName');
    }

    public function testGetNewReportBuilder() {
        $logoConfig = new LogoConfig();
        $dataProvider = new ReportStrategy(
            $this->fileNameProvider
        );
        $this->assertInstanceOf(ReportBuilder::class, $dataProvider->getNewReportBuilder([], $logoConfig, "test.html"));
    }

    public function testGetNewLogoConfig() {
        $dataProvider = new ReportStrategy(
            $this->fileNameProvider
        );
        $this->assertInstanceOf(LogoConfig::class, $dataProvider->getNewLogoConfig());
    }

    public function testGetNewSplFileObject() {
        $dataProvider = new ReportStrategy(
            $this->fileNameProvider
        );
        $this->assertInstanceOf(\SplFileObject::class, $dataProvider->getNewSplFileObject("tests/reports/reportTest"));
    }

    public function testDefaultUsage() {
        $rows = array(
            array("name" => "one", "value" => 1),
            array("name" => "two", "value" => 2)
        );

        $this->logoConfig->expects($this->never())
            ->method("setClientLogo");

        $this->logoConfig->expects($this->never())
            ->method("setEnterpriseLogo");

        $this->logoConfig->expects($this->never())
            ->method("setProductLogo");

        $this->fileNameProvider->expects($this->once())
            ->method("generateFileName")
            ->with(DataExporter::EXPORT_FORMAT_REPORT)
            ->willReturn("fileName");

        $this->reportBuilder->expects($this->once())
            ->method("buildReport");

        $this->assertEquals(DataExporter::EXPORT_FORMAT_REPORT, $this->exportStrategy->formatName());
        $this->exportStrategy->start($this->metaData);
        $this->exportStrategy->writeRows($rows);
        $fileName = $this->exportStrategy->finish();
        $this->assertEquals($fileName, "fileName");
    }

    public function testUsageWithLogoConfiguration() {
        $this->metaData["clientLogo"] = "clientLogo";
        $this->metaData["enterpriseLogo"] = "enterpriseLogo";
        $this->metaData["productLogo"] = "productLogo";
        $rows = array(
            array("name" => "one", "value" => 1),
            array("name" => "two", "value" => 2)
        );

        $this->logoConfig->expects($this->once())
            ->method("setClientLogo")
            ->with("clientLogo");

        $this->logoConfig->expects($this->once())
            ->method("setEnterpriseLogo")
            ->with("enterpriseLogo");

        $this->logoConfig->expects($this->once())
            ->method("setProductLogo")
            ->with("productLogo");

        $this->fileNameProvider->expects($this->once())
            ->method("getFullFilePath")
            ->with("fileName");

        $this->fileNameProvider->expects($this->once())
            ->method("generateFileName")
            ->with(DataExporter::EXPORT_FORMAT_REPORT)
            ->willReturn("fileName");

        $this->reportBuilder->expects($this->once())
            ->method("buildReport");

        $this->assertEquals(DataExporter::EXPORT_FORMAT_REPORT, $this->exportStrategy->formatName());
        $this->exportStrategy->start($this->metaData);
        $this->exportStrategy->writeRows($rows);
        $fileName = $this->exportStrategy->finish();
        $this->assertEquals($fileName, "fileName");
    }

    public function testExceptionWhenMetadaIsInvalid() {
        $metaData = array(
            "columns" => array()
        );
        $this->expectException(Exception::class);
        $this->exportStrategy->start($metaData);
    }
}
