<?php
namespace Zeedhi\DataExporter\Service\DataProvider;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception\InvalidFieldNameException;
use Doctrine\DBAL\Query\QueryBuilder;
use Doctrine\ORM\Query\ParameterTypeInferer;
use Zeedhi\DataExporter\Service\DataProvider;
use Zeedhi\Framework\DataSource\Configuration;
use Zeedhi\Framework\DataSource\FilterCriteria;
use Zeedhi\Framework\DataSource\Manager\Exception;
use Zeedhi\Framework\DataSource\ParameterBag;
use Zeedhi\Framework\DataSource\AssociatedWithDataSource;
use Zeedhi\Framework\DataSource\Manager\Doctrine\NameProvider;
use Zeedhi\Framework\DataSource\Operator\DefaultOperator;
use Zeedhi\Framework\DBAL\Driver\OCI8\OCI8Statement;

/**
 * Manages the connection with SQL databases.
 */
class SQLConnection implements DataProvider{

    /** @var NameProvider $nameProvider The name provider for the connection. */
    protected $nameProvider;
    /** @var ParameterBag $parameterBag The parameters of the connection. */
    protected $parameterBag;
    /** @var Configuration $dataSourceConfig The configuration of the datasource. */
    protected $dataSourceConfig;
    /** @var Connection $connection The connection. */
    protected $connection;
    /** @var OCI8Statement $stmt The statement to be executed. */
    protected $stmt;
    /** @var Number $pageSize The size of the page. */
    protected $pageSize;

    /**
     * Constructor
     * @method
     * 
     * @param $connection Connection The connection.
     * @param $nameProvider NameProvider The name provider for the connection.
     * @param $parameterBag ParameterBag The parameters of the connection.
     */
    public function __construct(Connection $connection, NameProvider $nameProvider, ParameterBag $parameterBag) {
        $this->connection = $connection;
        $this->nameProvider = $nameProvider;
        $this->parameterBag = $parameterBag;
    }

    /**
     * Loads the current configured datasource with a filter criteria associated.
     * 
     * @param AssociatedWithDataSource $associatedWithDataSource The filter criteria that will be used to load.
     */
    protected function loadCurrentDataSource(AssociatedWithDataSource $associatedWithDataSource) {
        $this->dataSourceConfig = $this->nameProvider->getDataSourceByName($associatedWithDataSource->getDataSourceName());
    }

    /**
     * Completes the params with the parameters from the current parameter bag.
     *
     * @param QueryBuilder $query The object used to build the query.
     * @param $params The parameters to be filled with data from parameter bag.
     *
     * @return mixed The parameters completed.
     */
    protected function completeParamsWithParameterBag(QueryBuilder $query, $params) {
        $matches = array();
        preg_match_all('/:([A-Za-z0-9_]*)/', $query->getSQL(), $matches);
        foreach ($matches[1] as $paramName) {
            if (!isset($params[$paramName])) {
                $params[$paramName] = $this->parameterBag->get($paramName);
            }
        }

        return $params;
    }

    /**
     * Create the query conditions based on a given filter criteria.
     *
     * @param FilterCriteria $filterCriteria The filter criteria that will be used to build the query.
     * @param QueryBuilder $query The query builder that will build the query.
     *
     * @return array The query built.
     */
    protected function buildQueryConditions(FilterCriteria $filterCriteria, QueryBuilder $query) {
        $params = array();
        foreach($filterCriteria->getConditions() as $condition) {
            DefaultOperator::factoryFromStringRepresentation($condition['operator'], $this->dataSourceConfig)
                ->addConditionToQuery($condition, $query, $params);
        }

        return $params;
    }

    /**
     * Creates the query builder for the current instance of the SQLConnection.
     * 
     * @return QueryBuilder The created instance of the query builder.
     */
    protected function createQueryBuilder() {
        return $this->connection->createQueryBuilder();
    }

    /**
     * Rethrows the exception.
     *
     * @param InvalidFieldNameException $e The exception to be rethrown.
     * @return Exception The rethrown exception.
     */
    protected function rethrowException(InvalidFieldNameException $e) {
        $matches = array();
        preg_match(": \"[A-Za-z_]+\":", $e->getPrevious()->getMessage(), $matches);
        $column = trim(current($matches), "\" ");
        return Exception::columnNotPresentInResultSet($column, $this->dataSourceConfig->getName(), $e);
    }

    /**
     * Infers the type of the params in a given object, returning as a compatible constant.
     *
     * @param $params The param whose params type should be infered.
     *
     * @return array An array containing the types of the params.
     */
    protected function inferTypes($params) {
        $types = array();
        foreach ($params as $name => $value) {
            $types[$name] = ParameterTypeInferer::inferType($value);
        }
        return $types;
    }

    /**
     * Adds the orderBy property to the query.
     * 
     * @param QueryBuilder $query The query builder that will build the query. 
     * @param array $orderBy An array containing the values that should be used to order by.
     */
    protected function addOrderByToQuery(QueryBuilder $query, $orderBy) {
        foreach ($orderBy as $columnName => $direction) {
            $query->addOrderBy($columnName, $direction);
        }
    }

    /**
     * Apply a new dataSource name from the filter criteria.
     *
     * @param FilterCriteria $filterCriteria The filter criteria that will be used.
     */
    protected function setDataSourceName(FilterCriteria $filterCriteria) {
        $this->dataSourceName = $filterCriteria->getDataSourceName();
    }


    /**
     * Apply on the query the orderBy's defined on the filter criteria.
     *
     * @param FilterCriteria $filterCriteria The filter criteria that will be used.
     * @param QueryBuilder $query The query builder that will build the query.
     */
    protected function processOrderBy(FilterCriteria $filterCriteria, QueryBuilder $query) {
        $this->addOrderByToQuery($query, $this->dataSourceConfig->getOrderBy());
        $this->addOrderByToQuery($query, $filterCriteria->getOrderBy());
    }

    /**
     * Add a grouping expression on the query.
     *
     * @param QueryBuilder $query The query builder that will build the query.
     * @param $groupBy The grouping expression.
     */
    protected function addGroupByToQuery(QueryBuilder $query, $groupBy) {
        $query->addGroupBy($groupBy);
    }

    /**
     * Apply on the query the grouping expressions defined on the filter criteria.
     *
     * @param FilterCriteria $filterCriteria The filter criteria that will be used.
     * @param QueryBuilder $query The query builder that will build the query.
     */
    protected function processGroupBy(FilterCriteria $filterCriteria, QueryBuilder $query) {
        $this->addGroupByToQuery($query, $this->dataSourceConfig->getGroupBy());
        $this->addGroupByToQuery($query, $filterCriteria->getGroupBy());
    }

    /**
     * Apply on the query the paginations defined on the filter criteria.
     *
     * @param FilterCriteria $filterCriteria The filter criteria that will be used.
     * @param QueryBuilder $query The query builder that will build the query.
     */
    protected function processPagination(FilterCriteria $filterCriteria, QueryBuilder $query) {
        if ($filterCriteria->isPaginated()) {
            $query->setFirstResult($filterCriteria->getFirstResult());
            $query->setMaxResults($filterCriteria->getPageSize());
        } else {
            if ($maxResults = $this->dataSourceConfig->getResultSetLimit()) {
                $query->setMaxResults($maxResults);
            }
        }
    }

    /**
     * Apply on the query the where clause defined on the filter criteria.
     *
     * @param FilterCriteria $filterCriteria The filter criteria that will be used.
     * @param QueryBuilder $query The query builder that will build the query.
     * @return array The query with the where clause set.
     */
    protected function processWhereClause(FilterCriteria $filterCriteria, QueryBuilder $query) {
        $params = $this->buildQueryConditions($filterCriteria, $query);
        if ($filterCriteria->hasWhereClause()) {
            $query->andWhere($filterCriteria->getWhereClause());
            foreach ($filterCriteria->getWhereClauseParams() as $bindColumnName => $value) {
                $bindColumnName = ltrim($bindColumnName, ':');
                $params[$bindColumnName] = $value;
            }
        }

        foreach ($this->dataSourceConfig->getConditions() as $condition) {
            $query->andWhere($condition);
        }

        $params = $this->completeParamsWithParameterBag($query, $params);
        return $params;
    }


    /**
     * Apply FilterCriteria conditions, whereClause and pagination to query.
     * Return a array of parameters needed to execute the query.
     *
     * @param FilterCriteria $filterCriteria The filter criteria that will be used.
     * @param QueryBuilder   $query
     * @return array the query with the FilterCriteria conditions set.
     */
    protected function processFilterConditions(FilterCriteria $filterCriteria, QueryBuilder $query) {
        $params = $this->processWhereClause($filterCriteria, $query);
        $this->processPagination($filterCriteria, $query);
        $this->processGroupBy($filterCriteria, $query);
        $this->processOrderBy($filterCriteria, $query);
        return $params;
    }

    /**
     * Creates the query to retrieve data from the dataSource.
     *
     * @return QueryBuilder The query created.
     */
    protected function createSelectQuery() {
        $select = $this->createQueryBuilder()->select($this->dataSourceConfig->getColumnsForResultSet());
        if($this->dataSourceConfig->hasQuery()) {
            $from = "(".$this->dataSourceConfig->getQuery().") ZEEDHI_ALIAS";
        } else {
            $from = $this->dataSourceConfig->getTableName();
        }
        $select->from($from);

        return $select;
    }

    /**
     * Builds the statement to be executed.
     *
     * @param FilterCriteria $filterCriteria The filter criteria that will be used.
     *
     * @throws
     */
    protected function buildStatement(FilterCriteria $filterCriteria) {
        try {
            $query = $this->createSelectQuery();
            $params = $this->processFilterConditions($filterCriteria, $query);
            $types = $this->inferTypes($params);
            $query->setParameters($params, $types);
            $this->stmt = $query->execute();
        } catch (InvalidFieldNameException $e) {
            throw $this->rethrowException($e);
        }
    }

    /**
     * Start a connection with database.
     *
     * @param FilterCriteria $filterCriteria The filter criteria that will be used.
     */

    public function start(FilterCriteria $filterCriteria) {
        $this->loadCurrentDataSource($filterCriteria);
        $this->buildStatement($filterCriteria);
        $this->setDataSourceName($filterCriteria);
        $this->pageSize = $filterCriteria->getPageSize();
    }

    /**
     * Return an array of rows that match the given stmt started.
     *
     * @return array The result of the filter criteria.
     */
    public function fetch() {
        $rows = array();
        $i = 0;

        while (($i < $this->pageSize) && ($row = $this->stmt->fetch())) {
            $rows[] = $row;
            $i++;
        }

        return $rows;
    }

    /**
     * Finishes the routine of providing the data.
     */
    public function finish()
    {
        $this->stmt->closeCursor();
    }

}