<?php
namespace Zeedhi\DataExporter\Controller;

use Zeedhi\DataExporter\Service\DataExporter as DataExporterService;
use Zeedhi\DataExporter\Utility\UniqueFileNameProvider;
use Zeedhi\DataExporter\Utility\ReportInfoManager;
use Zeedhi\Framework\DataSource\Configuration;
use Zeedhi\Framework\DataSource\FilterCriteria;
use Zeedhi\Framework\DataSource\Manager\Doctrine\NameProvider;
use Zeedhi\Framework\DataSource\ParameterBag;
use Zeedhi\Framework\DTO\Exception;
use Zeedhi\Framework\DTO\Request;
use Zeedhi\Framework\DTO\Response;
use Zeedhi\Framework\HTTP\Response\FileDownload;

class DataExporterTest extends \PHPUnit_Framework_TestCase {

    /** @var ReportGenerator */
    private $reportGenerator;
    /** @var Filter|\PHPUnit_Framework_MockObject_MockObject */
    private $request;
    /** @var Response|\PHPUnit_Framework_MockObject_MockObject */
    private $response;
    /** @var ParameterBag|\PHPUnit_Framework_MockObject_MockObject */
    private $parameterBag;
    /** @var NameProvider|\PHPUnit_Framework_MockObject_MockObject */
    private $dataSourceNameProvider;
    /** @var DataExporterService|\PHPUnit_Framework_MockObject_MockObject */
    private $dataExporterService;
    /** @var UniqueFileNameProvider|\PHPUnit_Framework_MockObject_MockObject  */
    private $uniqueFileNameProvider;
    /** @var integer */
    private $originalTimeout;
    /** @var integer */
    private $timeout;
    /** @var array */
    private $expectedMetaData;
    /** @var ReportInfoManager */
    private $reportInfoManager;

    protected function setUp() {
        $this->request = $this->getMockBuilder(Request\Filter::class)
            ->disableOriginalConstructor()
            ->setMethods(array(
                "getParameters",
                "populateParameterBag",
                "getFilterCriteria"
            ))
            ->getMock();

        $this->response = $this->getMockBuilder(Response::class)
            ->setMethods(array("addMessage", "setError", "setFileToDownload"))
            ->getMock();

        $this->parameterBag = $this->getMockBuilder(ParameterBag::class)
            ->disableOriginalConstructor()
            ->setMethods(array("get", "set"))
            ->getMock();

        $this->dataSourceNameProvider = $this->getMockBuilder(NameProvider::class)
            ->getMock();

        $this->dataExporterService = $this->getMockBuilder(DataExporterService::class)
            ->disableOriginalConstructor()
            ->setMethods(array("createExportFile"))
            ->getMock();

        $this->uniqueFileNameProvider = $this->getMockBuilder(UniqueFileNameProvider::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->reportInfoManager = $this->getMockBuilder(ReportInfoManager::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->originalTimeout = ini_get('max_execution_time');
        $this->timeout = 10;

        $this->reportGenerator = new ReportGenerator(
            $this->parameterBag,
            $this->dataSourceNameProvider,
            $this->dataExporterService,
            $this->uniqueFileNameProvider,
            $this->reportInfoManager,
            $this->timeout
        );

        $this->expectedMetaData = array(
            "dbType" => "SQL",
            "columns" => array(
                "value" => array(
                    "size"        => "30%",
                    "align"       => "left",
                    "sequence"    => 2,
                    "description" => "Value"
                ),
                "name" => array(
                    "size"        => "70%",
                    "align"       => "left",
                    "sequence"    => 1,
                    "description" => "Name"
                )
            )
        );
    }

    public function tearDown() {
        ini_set('max_execution_time', $this->originalTimeout);
    }

    public function testMissingMetadata() {
        $this->request->expects($this->once())
            ->method("getParameters")
            ->willReturn(array('dataSourceName' => "dataSourceName"));

        $this->response->expects($this->once())
            ->method("setError")
            ->with($this->callback(function($error) {
                return $error instanceof Response\Error
                    && $error->getMessage() === "A unexpected error occurred: Undefined index: metaData";
            }));

        $this->reportGenerator->createDataExportXLSFile($this->request, $this->response);
    }

    public function mockGetParameter() {
        $this->request->expects($this->exactly(1))
            ->method("getParameters")
            ->willReturn(array("metaData" => $this->expectedMetaData, 'dataSourceName' => "dataSourceName"));
    }

    public function testEmptyConditionsNoQuery() {
        $this->mockGetParameter();
        $expectedConditions = array(
            array(
                "columnName" => "value",
                "operator" => FilterCriteria::GTE,
                "value" => 1,
            )
        );

        $this->request->expects($this->once())
            ->method("getFilterCriteria")
            ->willReturn(new FilterCriteria("", $expectedConditions));

        $dataSourceConfiguration = $this->getMockBuilder(Configuration::class)
            ->disableOriginalConstructor()
            ->setMethods(array("getQuery"))
            ->getMock();

        $this->dataSourceNameProvider->expects($this->once())
            ->method("getDataSourceByName")
            ->with("dataSourceName")
            ->willReturn($dataSourceConfiguration);

        $dataSourceConfiguration->expects($this->once())
            ->method("getQuery")->willReturn(null);

        $this->dataExporterService->expects($this->once())
            ->method("createExportFile")
            ->with(
                $this->callback(function($filterCriteria) use ($expectedConditions) {
                    return $filterCriteria instanceof FilterCriteria
                        && $filterCriteria->getDataSourceName() === "dataSourceName"
                        && $filterCriteria->getConditions() === $expectedConditions;
                }),
                $this->expectedMetaData
            )->willReturn("exportedFileName.xls");


        $this->reportGenerator->createDataExportXLSFile($this->request, $this->response);
    }

    public function testExportWithParameterBag(){
        $this->mockGetParameter();
        $expectedConditions = array(
            array(
                "columnName" => "value",
                "operator" => FilterCriteria::GTE,
                "value" => 1,
            ),
            array(
                "columnName" => "column2",
                "operator" => FilterCriteria::EQ,
                "value" => 2,
            ),
        );

        $this->request->expects($this->once())
            ->method("getFilterCriteria")
            ->willReturn(new FilterCriteria("", $expectedConditions));

        $dataSourceConfiguration = $this->getMockBuilder(Configuration::class)
            ->disableOriginalConstructor()
            ->setMethods(array("getQuery"))
            ->getMock();

        $this->dataSourceNameProvider->expects($this->once())
            ->method("getDataSourceByName")
            ->with("dataSourceName")
            ->willReturn($dataSourceConfiguration);

        $dataSourceConfiguration->expects($this->once())
            ->method("getQuery")
            ->willReturn("SELECT * FROM TABLE_NAME WHERE value >= :value");

        $this->parameterBag->expects($this->once())->method("set")->with("value", 1);

        $this->dataExporterService->expects($this->once())
            ->method("createExportFile")
            ->with(
                $this->callback(function($filterCriteria) {
                    return $filterCriteria instanceof FilterCriteria
                    && $filterCriteria->getDataSourceName() === "dataSourceName"
                    && $filterCriteria->getConditions() === array(array(
                        "columnName" => "column2",
                        "operator" => FilterCriteria::EQ,
                        "value" => 2,
                    ));
                }),
                $this->expectedMetaData
            )->willReturn("exportedFileName.xls");

        $this->response->method("addMessage")
            ->with($this->callback(function($message) {
                return $message instanceof Response\Message
                    && $message->getMessage() === "exportedFileName.xls";
            }));

        $this->reportGenerator->createDataExportXLSFile($this->request, $this->response);
    }

    public function testExceptionOnService() {
        $this->mockGetParameter();
        $expectedConditions = array(
            array(
                "columnName" => "value",
                "operator" => FilterCriteria::GTE,
                "value" => 1,
            )
        );

        $this->request->expects($this->once())
            ->method("getFilterCriteria")
            ->willReturn(new FilterCriteria("", $expectedConditions));

        $dataSourceConfiguration = $this->getMockBuilder(Configuration::class)
            ->disableOriginalConstructor()
            ->setMethods(array("getQuery"))
            ->getMock();

        $this->dataSourceNameProvider->expects($this->once())
            ->method("getDataSourceByName")
            ->with("dataSourceName")
            ->willReturn($dataSourceConfiguration);

        $dataSourceConfiguration->expects($this->once())
            ->method("getQuery")->willReturn(null);

        $this->dataExporterService->expects($this->once())
            ->method("createExportFile")
            ->with(
                $this->callback(function($filterCriteria) use ($expectedConditions) {
                    return $filterCriteria instanceof FilterCriteria
                    && $filterCriteria->getDataSourceName() === "dataSourceName"
                    && $filterCriteria->getConditions() === $expectedConditions;
                }),
                $this->expectedMetaData
            )->willThrowException(\Zeedhi\DataExporter\Exception\Exception::nonAvailableExportFormat("pdf"));

        $this->response->expects($this->once())
            ->method("setError")
            ->with($this->callback(function($error) {
                return $error instanceof Response\Error
                && $error->getMessage() === "A error occurred while trying to generate your file: There are no strategies for given export format (pdf).";
            }));

        $this->reportGenerator->createDataExportXLSFile($this->request, $this->response);
    }

    public function testUnexpectedException() {
        $this->mockGetParameter();
        $expectedConditions = array(
            array(
                "columnName" => "value",
                "operator" => FilterCriteria::GTE,
                "value" => 1,
            )
        );

        $this->request->expects($this->once())
            ->method("getFilterCriteria")
            ->willReturn(new FilterCriteria("", $expectedConditions));

        $dataSourceConfiguration = $this->getMockBuilder(Configuration::class)
            ->disableOriginalConstructor()
            ->setMethods(array("getQuery"))
            ->getMock();

        $this->dataSourceNameProvider->expects($this->once())
            ->method("getDataSourceByName")
            ->with("dataSourceName")
            ->willReturn($dataSourceConfiguration);

        $dataSourceConfiguration->expects($this->once())
            ->method("getQuery")->willReturn(null);

        $this->dataExporterService->expects($this->once())
            ->method("createExportFile")
            ->with(
                $this->callback(function($filterCriteria) use ($expectedConditions) {
                    return $filterCriteria instanceof FilterCriteria
                        && $filterCriteria->getDataSourceName() === "dataSourceName"
                        && $filterCriteria->getConditions() === $expectedConditions;
                }),
                $this->expectedMetaData
            )->willThrowException(new \Exception("Unexpected exception."));

        $this->response->expects($this->once())
            ->method("setError")
            ->with($this->callback(function($error) {
                return $error instanceof Response\Error
                    && $error->getMessage() === "A unexpected error occurred: Unexpected exception.";
            }));

        $this->reportGenerator->createDataExportPDFFile($this->request, $this->response);
    }

    public function testTimeoutConfiguration(){
        $this->mockGetParameter();
        $expectedConditions = array(
            array(
                "columnName" => "value",
                "operator" => FilterCriteria::GTE,
                "value" => 1,
            )
        );

        $this->request->expects($this->once())
            ->method("getFilterCriteria")
            ->willReturn(new FilterCriteria("", $expectedConditions));

        $dataSourceConfiguration = $this->getMockBuilder(Configuration::class)
            ->disableOriginalConstructor()
            ->setMethods(array("getQuery"))
            ->getMock();

        $this->dataSourceNameProvider->expects($this->once())
            ->method("getDataSourceByName")
            ->with("dataSourceName")
            ->willReturn($dataSourceConfiguration);

        $dataSourceConfiguration->expects($this->once())
            ->method("getQuery")->willReturn(null);

        $this->dataExporterService->expects($this->once())
            ->method("createExportFile")
            ->with(
                $this->callback(function($filterCriteria) use ($expectedConditions) {
                    return $filterCriteria instanceof FilterCriteria
                        && $filterCriteria->getDataSourceName() === "dataSourceName"
                        && $filterCriteria->getConditions() === $expectedConditions;
                }),
                $this->expectedMetaData
            )->willReturn("exportedFileName.xls");


        $this->reportGenerator->createDataExportXLSFile($this->request, $this->response);

        $this->assertNotEquals($this->timeout, $this->originalTimeout);
    }

    public function testDownloadPDFExportedFile(){
        $uniqueFileNameProvider = new UniqueFileNameProvider("../reports");
        $dataExporter = new ReportGenerator(
            $this->parameterBag,
            $this->dataSourceNameProvider,
            $this->dataExporterService,
            $uniqueFileNameProvider,
            $this->reportInfoManager,
            $this->timeout,
            array("pdf" => "application/pdf")
        );
        $reportName = "PDFTest.pdf";
        $reportPath = $uniqueFileNameProvider->getFullFilePath($reportName);
        $request = new Request("POST", "download", "userId");
        $request->setParameter("reportName", $reportName);
        $expectedFile = new Response\File($reportPath, true, $reportName, "application/pdf");
        $this->response->expects($this->once())
            ->method("setFileToDownload")
            ->with($expectedFile);
        $dataExporter->downloadExportedFile($request, $this->response);
    }

    public function testDownloadXLSXExportedFile(){
        $uniqueFileNameProvider = new UniqueFileNameProvider("../reports");
        $dataExporter = new ReportGenerator(
            $this->parameterBag,
            $this->dataSourceNameProvider,
            $this->dataExporterService,
            $uniqueFileNameProvider,
            $this->reportInfoManager,
            $this->timeout
        );
        $reportName = "XLSXTest.xlsx";
        $reportPath = $uniqueFileNameProvider->getFullFilePath($reportName);
        $request = new Request("POST", "download", "userId");
        $request->setParameter("reportName", $reportName);
        $expectedFile = new Response\File($reportPath, true, $reportName, "application/xlsx");
        $this->response->expects($this->once())
            ->method("setFileToDownload")
            ->with($expectedFile);
        $dataExporter->downloadExportedFile($request, $this->response);
    }

    public function testFailedDownloadExportedFile(){
        $uniqueFileNameProvider = new UniqueFileNameProvider("../reports");
        $dataExporter = new ReportGenerator(
            $this->parameterBag,
            $this->dataSourceNameProvider,
            $this->dataExporterService,
            $uniqueFileNameProvider,
            $this->reportInfoManager,
            $this->timeout
        );
        $request = new Request("POST", "download", "userId");
        $request->setParameter("reportName", "Test.ext");
        $this->response->expects($this->never())
            ->method("setFileToDownload");
        $this->response->expects($this->once())
            ->method("setError")
            ->with($this->callback(function($error) {
                return $error instanceof Response\Error
                    && $error->getMessage() === "Unsupported report file extension";
            }));
        $dataExporter->downloadExportedFile($request, $this->response);
    }

    public function testDownloadHTMLExportedFile(){
        $uniqueFileNameProvider = new UniqueFileNameProvider("../reports");
        $dataExporter = new ReportGenerator(
            $this->parameterBag,
            $this->dataSourceNameProvider,
            $this->dataExporterService,
            $uniqueFileNameProvider,
            $this->reportInfoManager,
            $this->timeout
        );
        $reportName = "PDFTest.html";
        $reportPath = $uniqueFileNameProvider->getFullFilePath($reportName);
        $request = new Request("POST", "download", "userId");
        $request->setParameter("reportName", $reportName);
        $expectedFile = new Response\File($reportPath, true, $reportName, "text/html");
        $this->response->expects($this->once())
            ->method("setFileToDownload")
            ->with($expectedFile);
        $dataExporter->downloadExportedFile($request, $this->response);
    }
}
