<?php
namespace Zeedhi\DataExporter\Service;

use Zeedhi\DataExporter\Service\ExportStrategy\SpreadSheet;
use Zeedhi\DataExporter\Utility\UniqueFileNameProvider;

class SpreadSheetTest extends \PHPUnit_Framework_TestCase {

    /** @var SpreadSheet */
    private $exportStrategy;
    /** @var array */
    private $metaData;
    /** @var \XLSXWriter|\PHPUnit_Framework_MockObject_MockObject */
    private $xlsxWriter;
    /** @var UniqueFileNameProvider|\PHPUnit_Framework_MockObject_MockObject */
    private $fileNameProvider;

    protected function setUp() {
        $this->xlsxWriter = $this->getMockBuilder(\XLSXWriter::class)
            ->setMethods(array(
                "writeSheetRow",
                "writeSheetHeader",
                "writeToFile"
            ))
            ->getMock();

        $this->fileNameProvider = $this->getMockBuilder(UniqueFileNameProvider::class)
            ->disableOriginalConstructor()
            ->setMethods(array("generateFileName", "getFullFilePath"))
            ->getMock();

        $this->metaData = array(
            "columns" => array(
                "value" => array(
                    "size"        => "30%",
                    "align"       => "left",
                    "sequence"    => 10,
                    "description" => "Value"
                ),
                "name" => array(
                    "size"        => "60%",
                    "align"       => "left",
                    "sequence"    => 2
                ),
                "type" => array(
                    "size"        => "10%",
                    "align"       => "left",
                    "sequence"    => 30,
                    "description" => "Type"
                )
            ),
            "reportName" => "report",
            "xlsDefaultType" => "xlsx"
        );

        $this->exportStrategy = $this->getMockBuilder(SpreadSheet::class)
            ->setConstructorArgs(array($this->fileNameProvider))
            ->setMethods(array("getNewXLSWriter"))
            ->getMock();

        $this->exportStrategy->method("getNewXLSWriter")->willReturn($this->xlsxWriter);
    }

    public function testGetNewXLSWriter() {
        $dataProvider = new SpreadSheet(
            $this->fileNameProvider
        );
        $this->assertInstanceOf(\XLSXWriter::class, $dataProvider->getNewXLSWriter());
    }

    public function testDefaultUsage() {
        $rows = array(
            array("name" => "one", "value" => 1),
            array("name" => "two", "value" => 2, "type" => "test"),
            array("name" => "three", "value" => 3),
            array("name" => "four", "value" => 4),
            array("name" => "five", "value" => 5),
        );

        $this->fileNameProvider->expects($this->once())
            ->method("generateFileName")
            ->with(DataExporter::EXPORT_FORMAT_SPREADSHEET)
            ->willReturn("fileName");

        $this->fileNameProvider->expects($this->once())
            ->method("getFullFilePath")
            ->with("fileName")
            ->willReturn("fullFilePath");

        $this->xlsxWriter->expects($this->once())
            ->method("writeSheetHeader")
            ->with(SpreadSheet::SHEET_NAME, array("Value" => "xlsx", "Type" => "xlsx"));

        $this->xlsxWriter->expects($this->exactly(5))
            ->method("writeSheetRow")
            ->withConsecutive(
                array(SpreadSheet::SHEET_NAME, array("one", 1, "")),
                array(SpreadSheet::SHEET_NAME, array("two", 2, "test")),
                array(SpreadSheet::SHEET_NAME, array("three", 3, "")),
                array(SpreadSheet::SHEET_NAME, array("four", 4, "")),
                array(SpreadSheet::SHEET_NAME, array("five", 5, ""))
            );

        $this->xlsxWriter->expects($this->once())
            ->method("writeToFile")
            ->with("fullFilePath");

        $this->assertEquals(DataExporter::EXPORT_FORMAT_SPREADSHEET, $this->exportStrategy->formatName());
        $this->exportStrategy->start($this->metaData);
        $this->exportStrategy->writeRows($rows);
        $this->exportStrategy->finish();
    }
}
