<?php
namespace Zeedhi\DataExporter\Exception;

/**
 * Throws the possible exceptions generating the report.
 */
class Exception extends \Exception {

    /** @const integer Error code for a eport format not avaible. */
    const NON_AVAILABLE_EXPORT_FORMAT = 1;

    /** @const integer Error code for when there is a error wile fetching the data. */
    const EXCEPTION_ON_FETCH_DATA = 2;

    /** @const integer Error code for when the given report name is not founded. */
    const REPORT_NOT_FOUND = 3;

    /** @const integer Error code for when there is a necessary property missing in the report generation. */
    const MISSING_PROPERTY = 4;

    /**
     * Exception throwed when there the format is not avaible.
     *
     * @param  string $format The format not avaible.
     *
     * @return static The exception.
     */
    public static function nonAvailableExportFormat($format) {
        return new static(
            "There are no strategies for given export format ({$format}).",
            static::NON_AVAILABLE_EXPORT_FORMAT
        );
    }

    /**
     * Exception throwed when there is a error while tring to fetch the data.
     *
     * @param  \Exception $previousException   The previous exception.
     *
     * @return static  The exception.
     */
    public static function exceptionWhileFetchingData(\Exception $previousException) {
        $previousMessage = $previousException->getMessage();
        return new static(
            "A exception occurred while we try to fetch the data: {$previousMessage}.",
            static::EXCEPTION_ON_FETCH_DATA,
            $previousException
        );
    }

    /**
     * Expection throwed when the there is a property missing on the metadata.
     *
     * @param  $propertyName Name of the property missing on the report metadata.
     *
     * @return static The exception.
     */
    public static function missingProperty($propertyName) {
        return new static(
            "It's missing value for property {$propertyName} in report metadata.",
            static::MISSING_PROPERTY
        );
    }

    public static function reportNotFound() {
        return new static(
            "Report not found.",
            static::REPORT_NOT_FOUND
        );
    }
}
