<?php

namespace Zeedhi\DataExporter\Service;

use Zeedhi\Framework\DataSource\FilterCriteria;
use Zeedhi\Framework\DataSource\DataSet;
use Zeedhi\DataExporter\Service\DataHandler;
use Zeedhi\DataExporter\Exception\Exception;
use Zeedhi\Report\LogoConfig;
use Zeedhi\Report\ReportBuilder;

/**
 * Service for generating the reports.
 */
class DataExporter {

    /** @const string Format to generate a xlsx file. */
    const EXPORT_FORMAT_SPREADSHEET = 'xlsx';

    /** @const string Format to generate a pdf file. */
    const EXPORT_FORMAT_REPORT = 'pdf';

    /** @const string Format to generate a csv file. */
    const EXPORT_FORMAT_CSV = 'csv';

    /** @const integer Page size to delimit the quantity of rows in one page on the reports. */
    const PAGE_SIZE = 50;

    /** @var ExportStrategy[] Defines the avaible configuration to generate the report. */
    private $exportStrategies;

    /** @var DataProvider[]  Provides data to the exporter. */
    private $dataProviderArr;
    
    /** @var DataHandler  Provides adequate handling to given data. */
    private $dataHanlder;

    /**
     * Contructor.
     *
     * @param array        $exportStrategies
     * @param DataProvider $dataProvider
     */
    public function __construct(array $exportStrategies, array $dataProviderArr, DataHandler $dataHandler) {
        $this->exportStrategies = $exportStrategies;
        $this->dataProviderArr = $dataProviderArr;
        $this->dataHandler = $dataHandler;
    }

    /**
     * Gets the export strategy.
     *
     * Get the given format and searches for the strategy related to it.
     *
     * @param string $format   Format that the report will be generated.
     *
     * @return ExportStrategy Strategy related to the given format.
     *
     * @throws Exception If the given format does not have a export strategy.
     */
    private function selectExportStrategy($format)
    {
        $exportStrategy = null;
        foreach ($this->exportStrategies as $candidateStrategy) {
            if ($candidateStrategy->formatName() === $format) {
                $exportStrategy = $candidateStrategy;
            }
        }

        if($exportStrategy === null) {
            throw Exception::nonAvailableExportFormat($format);
        }
        return $exportStrategy;
    }

    private function selectDataProvider($type){
        return $this->dataProviderArr[$type];
    }

    protected function validateMetaData($reportMetaData) {
        static $propertiesToValidate = array(
            "title"
        );
        foreach($propertiesToValidate as $propertyName) {
            if (!isset($reportMetaData[$propertyName])) {
                throw Exception::missingProperty($propertyName);
            }
        }
    }

    /**
     * Create the export file.
     *
     * Read the data provider with the given FilterCriteria and dump the data to a file of specified data format.
     *
     * @param FilterCriteria $filterCriteria
     * @param array $metaData
     * @param string $format
     *
     * @return string The name of the file with dumped data.
     */
    public function createExportFile(FilterCriteria $filterCriteria, array $metaData, $format) {
        $exportStrategy = $this->selectExportStrategy($format);
        $dataProvider = $this->selectDataProvider($metaData['dbType']);
        $exportStrategy->start($metaData);
        $filterCriteria->setPageSize(self::PAGE_SIZE);
        $dataProvider->start($filterCriteria);
        do {
            $rows = $dataProvider->fetch();
            if (count($rows) > 0) {
                $exportStrategy->writeRows($this->decodeRows($metaData, $rows));
            }
        } while (count($rows) == self::PAGE_SIZE);
        $dataProvider->finish();
        $fileName = $exportStrategy->finish();

        return $fileName;
    }

    /**
     * Create the export file.
     *
     * Read the data provider with the given FilterCriteria and dump the data to a file of specified data format.
     *
     * @param FilterCriteria $filterCriteria
     * @param array $metaData
     * @param string $format
     *
     * @return string The name of the file with dumped data.
     */
    public function decodeRows(array $metaData, array $dataSet){
        $staticData =  isset($metaData["staticData"]) ? $metaData["staticData"] : array();
        return $this->dataHandler->decodeRows($dataSet, $staticData);
    }
    
    /**
     * Order dataSet using field groups and order by.
     *
     * @param array $metaData
     * @param array $metaData
     *
     * @return string The name of the file with dumped data.
     */
    public function orderData(array $conditions, array $dataSet, $metaData = []){
        foreach ($dataSet as $key=>$value) {
            if(isset($dataSet[$key]["__groupHeaders"])) unset($dataSet[$key]["__groupHeaders"]);
        }
        
        $orderBy = [];
        foreach ($conditions as $condition) {
            if(is_array($condition["name"])){
                $orderBy = array_merge($orderBy, $condition["name"]);
            }
        }

        return $this->dataHandler->orderData($orderBy, $dataSet, $metaData);
    }

    /**
     * Create the report.
     *
     * Dump the data given to a file of specified data format.
     *
     * @param array  $metaData
     * @param array  $dataSet
     * @param string $format
     *
     * @return string The name of the file with dumped data.
     */
    public function generateReport($metaData, $dataSet, $format) {
        $exportStrategy = $this->selectExportStrategy($format);
        $exportStrategy->start($metaData);
        $decodedDataSet = $this->decodeRows($metaData, $dataSet);
        $exportStrategy->writeRows($decodedDataSet);
        $fileName = $exportStrategy->finish();
        return $fileName;
    }


}
