<?php
namespace Zeedhi\DataExporter\Service\ExportStrategy;

use Zeedhi\DataExporter\Exception\Exception;
use Zeedhi\DataExporter\Service\DataExporter;
use Zeedhi\DataExporter\Service\ExportStrategy;
use Zeedhi\DataExporter\Utility\UniqueFileNameProvider;

use Zeedhi\Report\LogoConfig;
use Zeedhi\Report\ReportBuilder;


/**
 * Manages the routone of generating a file.
 */
class ReportStrategy implements ExportStrategy {

    /** @const string Default path to the report's templates. */
    const DEFAULT_TEMPLATES_PATH = "/../../../../report/templates/";

    /** @var resource File in wich will be created the report.*/
    private $file;

    /** @var UniqueFileNameProvider Provides name and path to the reports. */
    private $fileNameProvider;

    /** @var String Path of the templates used by the report. */
    private $templatePath;

    /** @var String Path of the report assets. */
    private $assetsPath;

    /** @var String Data to be written on the file after all report's data being written. */
    private $finalPart;

    private $builder;



    /**
     * Constructor.
     *
     * @param UniqueFileNameProvider $fileNameProvider
     * @param string                 $templatePath
     * @param string                 $assetsPath
     */
    public function __construct(UniqueFileNameProvider $fileNameProvider, $templatePath = null, $assetsPath = './assets'){
        $this->templatePath = $templatePath ?: realpath(__DIR__.self::DEFAULT_TEMPLATES_PATH);
        $this->assetsPath = $assetsPath;
        $this->fileNameProvider = $fileNameProvider;
    }

    /**
     * Get a report builder.
     *
     * Retrives an ReportBuilder constructed with the paams and the templatePath and assetsPath declared at the constructor.
     *
     * @param  array      $metaData   Description of the report's structure.
     * @param  array      $gridData   Data of the grid in wich the reports will be generated.
     * @param  LogoConfig $logoConfig Defines the description of the report.
     * @param  string     $fileName   The file name,
     *
     * @return ReportBuilder A ReportBuild.
     */
    public function getNewReportBuilder(array $metaData, LogoConfig $logoConfig, $fileName){
        return new ReportBuilder($metaData, [], $logoConfig, null, null, $fileName, null);
    }

    /**
     * Get a LogoConfig
     *
     * Retrieves a intance of LogoConfig.
     *
     * @return LogoConfig Defines the description of the report.
     */
    public function getNewLogoConfig(){
        return new LogoConfig();
    }

    /**
     * Get a SplFileObject with the given name and option.
     *
     * @param  string $fileName Name of the file in which the file object will be created.
     * @param  string $option   Set the option in with the file will be manipulated. Can be "r", "w" or "x".
     *
     * @return \SplFileObject A SplFileObject to manipulate a file according to the given configuration.
     */
    public function getNewSplFileObject($fileName, $option = "r"){
        return new \SplFileObject($fileName, $option);
    }

    /**
     * Validate the given metadata.
     *
     * Validate the given metadata by getting the static properties to validate, defined an a inside variables
     * and testing if the properties are setted.
     *
     * @param Object $reportMetaData Description of the report structure.
     *
     * @throws Exception Is the property is not valid.
     */
    protected function validateMetaData($reportMetaData) {
        static $propertiesToValidate = array(
            "title"
        );
        foreach($propertiesToValidate as $propertyName) {
            if (!isset($reportMetaData[$propertyName])) {
                throw Exception::missingProperty($propertyName);
            }
        }
    }

    /**
     * Get the ReportLogoConfig.
     *
     * Get a ReportLogoConfig configurated based on the given metadata.
     *
     * @param  Object $metaData Description of the report's structure.
     *
     * @return LogoConfig
     */
    protected function getLogoConfig($metaData){
        $logoConfig = $this->getNewLogoConfig();
        if (isset($metaData['clientLogo'])) {
            $logoConfig->setClientLogo($metaData['clientLogo']);
        }

        if (isset($metaData['enterpriseLogo'])) {
            $logoConfig->setEnterpriseLogo($metaData['enterpriseLogo']);
        }

        if (isset($metaData['productLogo'])) {
            $logoConfig->setProductLogo($metaData['productLogo']);
        }
        return $logoConfig;
    }

    /**
     * Start to generate the report.
     *
     * Gets filename and path, validate the metadata, get a new SplFileObject and writes the header's report on the file.
     *
     * @param  array  $metaData Description of the report's strucuture.
     */
    public function start(array $metaData){
        $this->validateMetaData($metaData);
        $reportName = $this->fileNameProvider->generateFileName($this->formatName());
        $targetFile = $this->fileNameProvider->getFullFilePath($reportName);
        $logoConfig = $this->getLogoConfig($metaData);

        $this->builder = $this->getNewReportBuilder($metaData, $logoConfig, $targetFile);
        $this->builder->buildReport("pdf");
    }

    /**
     * Write the given rows on the file.
     *
     * Encode the given array, removes the json's opening and close bracket, and write the file
     *
     * @param  array $rows Rows to be written on the file.
     */
    public function writeRows(array $rows){
        $this->builder->writeRows($rows);
    }

    /**
     * Finishes the report generation.
     *
     * Finishes the report generation by wtitten the bottom of the file and returning the name of the
     * generated file.
     *
     * @return string Name of the generated file.
     */
    public function finish(){
        return $this->builder->finish();
    }

    /**
     * Get the file format of the report strategy.
     *
     * @return string The format of the report file.
     */
    public function formatName(){
        return DataExporter::EXPORT_FORMAT_REPORT;
    }
}