<?php
namespace Zeedhi\DataExporter\Service;

use Zeedhi\DataExporter\Exception\Exception;
use Zeedhi\DataExporter\Service\DataHandler\StaticData;

class DataExporterTest extends \PHPUnit_Framework_TestCase {

    /** @var DataExporter */
    private $dataExporter;
    /** @var DataProvider|\PHPUnit_Framework_MockObject_MockObject */
    private $dataProvider;
    /** @var ExportStrategy|\PHPUnit_Framework_MockObject_MockObject  */
    private $exportStrategy;
    private $dataHandler;
    /** @var \Zeedhi\Framework\DataSource\FilterCriteria */
    private $filterCriteria;
    /** @var array */
    private $metaData;

    protected function setUp() {
        $this->dataProvider = $this->getMockBuilder(DataProvider::class)
            ->getMock();

        $this->exportStrategy = $this->getMockBuilder(ExportStrategy::class)
            ->getMock();

        $this->dataHandler = $this->getMockBuilder(StaticData::class)
            ->getMock();

        $this->exportStrategy->expects($this->once())->method('formatName')->willReturn(DataExporter::EXPORT_FORMAT_CSV);

        $this->dataExporter = new DataExporter(
            array($this->exportStrategy),
            array("SQL" => $this->dataProvider),
            $this->dataHandler
        );

        $this->filterCriteria = new \Zeedhi\Framework\DataSource\FilterCriteria("dataSourceName");
        $this->metaData = array(
            "dbType" => "SQL",
            "columns" => array(
                array("name" => "name", "label" => "Name"),
                array("name" => "value", "label" => "Value")
            )
        );
    }

    private function createDataProviderMocks(\PHPUnit_Framework_MockObject_MockObject $dataProvider, $rows) {
        $dataProvider->expects($this->once())->method('start');
        $dataProvider->expects($this->once())
            ->method('fetch')
            ->willReturn($rows);
        $dataProvider->expects($this->once())->method('finish');
    }

    public function testDefaultUse() {
        $rows = array(
            array("name" => "one", "value" => 1),
            array("name" => "two", "value" => 2),
            array("name" => "three", "value" => 3),
            array("name" => "four", "value" => 4),
            array("name" => "five", "value" => 5),
        );

        $this->createDataProviderMocks($this->dataProvider, $rows);

        $this->exportStrategy->expects($this->once())->method("start")->with($this->metaData);
        $this->exportStrategy->expects($this->once())->method("writeRows")->with($rows);
        $this->exportStrategy->expects($this->once())->method("finish")->willReturn("fileName.csv");

        $this->dataHandler->expects($this->once())->method("decodeRows")->willReturn($rows);

        $fileName = $this->dataExporter->createExportFile(
            $this->filterCriteria,
            $this->metaData,
            DataExporter::EXPORT_FORMAT_CSV
        );
        $this->assertEquals("fileName.csv", $fileName);
    }

    public function testSelectCorrectOutputStrategy() {
        $rows = array(
            array("name" => "one", "value" => 1),
            array("name" => "two", "value" => 2),
            array("name" => "three", "value" => 3),
            array("name" => "four", "value" => 4),
            array("name" => "five", "value" => 5),
        );

        $this->createDataProviderMocks($this->dataProvider, $rows);

        $wrongOutputStrategy = $this->getMockBuilder(ExportStrategy::class)
            ->getMock();

        $wrongOutputStrategy->expects($this->once())->method('formatName')->willReturn("wrong");

        $this->exportStrategy->expects($this->once())->method("start")->with($this->metaData);
        $this->exportStrategy->expects($this->once())->method("writeRows")->with($rows);
        $this->exportStrategy->expects($this->once())->method("finish")->willReturn("fileName.csv");

        $this->dataHandler->expects($this->once())->method("decodeRows")->willReturn($rows);

        $dataExporter = new DataExporter(
            array($wrongOutputStrategy, $this->exportStrategy),
            array("SQL" => $this->dataProvider),
            $this->dataHandler
        );

        $fileName = $dataExporter->createExportFile(
            $this->filterCriteria,
            $this->metaData,
            DataExporter::EXPORT_FORMAT_CSV
        );
        $this->assertEquals("fileName.csv", $fileName);
    }

    public function testNonAvailableExportStrategy() {
        $this->expectException(Exception::class);
        $this->expectExceptionMessage("There are no strategies for given export format (xlsx).");
        $this->expectExceptionCode(Exception::NON_AVAILABLE_EXPORT_FORMAT);
        $this->dataExporter->createExportFile(
            $this->filterCriteria,
            $this->metaData,
            DataExporter::EXPORT_FORMAT_SPREADSHEET
        );
    }

    public function testFetchException() {
        $this->dataProvider->expects($this->once())->method('start');

        $expectedException = Exception::exceptionWhileFetchingData(new \Exception("Previous Message"));
        $this->dataProvider->expects($this->once())
            ->method('fetch')
            ->will($this->throwException($expectedException));

        $this->exportStrategy->expects($this->once())->method("start")->with($this->metaData);

        $this->expectException(Exception::class);
        $this->expectExceptionMessage("A exception occurred while we try to fetch the data: Previous Message");
        $this->expectExceptionCode(Exception::EXCEPTION_ON_FETCH_DATA);
        $this->dataExporter->createExportFile(
            $this->filterCriteria,
            $this->metaData,
            DataExporter::EXPORT_FORMAT_CSV
        );
    }

    private function factoryPage($currentPage, $pageSize) {
        $pagesRow = array();
        for ($rowId = 0; $rowId < $pageSize; $rowId++) {
            $value = $currentPage * $pageSize + $rowId;
            $pagesRow[] = array("name" => "row {$value}", "value" => $value);
        }

        return $pagesRow;
    }

    public function factoryParamsValidation($currentPage) {
        return array($this->callback(function ($filterCriteria) use ($currentPage) {
            return $filterCriteria instanceof \Zeedhi\Framework\DataSource\FilterCriteria
                && $filterCriteria->getDataSourceName() === "dataSourceName"
                // Line above commented because current phpunit has a bug while evaluating consecutiveParameters
                // && $filterCriteria->getPage() === $currentPage
                && $filterCriteria->getPageSize() === 50;
        }));
    }

    public function testMultiplesPages() {
        $page1 = $this->factoryPage(1, 50);
        $page2 = $this->factoryPage(2, 50);
        $page3 = $this->factoryPage(3, 50);
        $page4 = array();

        $fetchParam1 = $this->factoryParamsValidation(1);
        $fetchParam2 = $this->factoryParamsValidation(2);
        $fetchParam3 = $this->factoryParamsValidation(3);
        $fetchParam4 = $this->factoryParamsValidation(4);

        $this->dataProvider->expects($this->once())->method('start');
        $this->dataProvider->expects($this->exactly(4))
            ->method('fetch')
            ->will($this->onConsecutiveCalls($page1, $page2, $page3, $page4));
        $this->dataProvider->expects($this->once())->method('finish');

        $this->exportStrategy->expects($this->once())->method("start")->with($this->metaData);
        $this->exportStrategy->expects($this->exactly(3))
            ->method("writeRows")
            ->withConsecutive(array($page1), array($page2), array($page3));
        $this->exportStrategy->expects($this->once())->method("finish")->willReturn("fileName.csv");

        $this->dataHandler->expects($this->exactly(3))
            ->method("decodeRows")
            ->will($this->onConsecutiveCalls($page1, $page2, $page3));

        $fileName = $this->dataExporter->createExportFile(
            $this->filterCriteria,
            $this->metaData,
            DataExporter::EXPORT_FORMAT_CSV
        );
        $this->assertEquals("fileName.csv", $fileName);
    }
}
