<?php
namespace Zeedhi\DataExporter\Controller;

use Zeedhi\DataExporter\Service\SpreadSheet\SpreadsheetWriter;
use Zeedhi\DataExporter\Service\DataExporter as DataExporterService;
use Zeedhi\DataExporter\Service\CSVService;
use Zeedhi\DataExporter\Exception\Exception as ServiceException;
use Zeedhi\DataExporter\Utility\UniqueFileNameProvider;
use Zeedhi\Framework\Controller\Simple;
use Zeedhi\Framework\DataSource\Configuration;
use Zeedhi\Framework\DataSource\FilterCriteria;
use Zeedhi\Framework\DataSource\ParameterBag;
use Zeedhi\Framework\DTO\Exception as DTOException;
use Zeedhi\Framework\DTO\Request;
use Zeedhi\Framework\DTO\Response;
use Zeedhi\Framework\DTO\Request\Row;
use Zeedhi\Framework\DataSource\Manager\Doctrine\NameProvider;
use Zeedhi\DataExporter\Utility\ReportInfoManager;
use Zeedhi\Framework\DependencyInjection\InstanceManager;

/**
 * Responsible for recieve the requests of generating reports and treat them.
 */
class ReportGenerator extends Simple {

    /** @var DataExporterService Service to generate the report. */
    private $dataExporter;

    /** @var NameProvider Get the datasourceConfiguration. */
    private $dataSourceNameProvider;

    /** @var ParameterBag Hold parameters for the filter. */
    private $parameterBag;

    /** @var UniqueFileNameProvider Provides name and path to the report. */
    private $fileNameProvider;

    /** @var integer Reconfigure the response timeout. This value is setted in seconds. */
    private $timeout;

    /** @var array Mapping of supported mime types by supported extensions */
    private $mimeTypeByExtension = array(
        "html" => "text/html",
        "xlsx" => "application/xlsx",
    );

    private $reportInfoManager;
    /**
     * @var SpreadsheetWriter
     */
    private $spreadsheetWriter;
    private $csvService;
    protected $dataSet;
    protected $metaData;
    protected $filter;

    /**
     * Constructor.
     *
     * @param ParameterBag           $parameterBag
     * @param NameProvider           $dataSourceNameProvider
     * @param DataExporterService    $dataExporter
     * @param UniqueFileNameProvider $fileNameProvider
     * @param ReportInfoManager      $reportInfoManager
     * @param SpreadsheetWriter      $spreadsheetWriter
     * @param integer                $timeout
     * @param array                  $supportedMimeTypesByExtension
     */
    public function __construct(
        ParameterBag $parameterBag,
        NameProvider $dataSourceNameProvider,
        DataExporterService $dataExporter,
        UniqueFileNameProvider $fileNameProvider,
        ReportInfoManager $reportInfoManager,
        SpreadsheetWriter $spreadsheetWriter,
        $timeout,
        $supportedMimeTypesByExtension = array()
    ) {
        $this->parameterBag = $parameterBag;
        $this->dataSourceNameProvider = $dataSourceNameProvider;
        $this->fileNameProvider = $fileNameProvider;
        $this->dataExporter = $dataExporter;
        $this->timeout = $timeout;
        $this->reportInfoManager = $reportInfoManager;
        $this->spreadsheetWriter = $spreadsheetWriter;
        $this->csvService = InstanceManager::getInstance()->getService("\Zeedhi\DataExporter\Service\CSVService");
        $this->mimeTypeByExtension = array_merge($this->mimeTypeByExtension, $supportedMimeTypesByExtension);
    }

    /**
     * Reset the timeout limit.
     *
     * Reset the using the timeout passed at the constructor.
     */
    private function startTimeout(){
        set_time_limit($this->timeout);
    }

    public static function getPlaceholderPositions(string $sql): array
    {
        $params = [];

        // remove strings para evitar falso positivo
        $sql = preg_replace("/'(?:''|[^'])*'/", '', $sql);
        $sql = preg_replace('/"(?:\\"|[^"])*"/', '', $sql);

        if (preg_match_all('/:([a-zA-Z_][a-zA-Z0-9_]*)/', $sql, $matches)) {
            foreach ($matches[1] as $name) {
                $params[] = $name;
            }
        }

        return array_unique($params);
    }

    /**
     * Get the conditions for the filterCriteria.
     *
     * Builds the conditions by searching on the datasourceConfig query if there is one, or on the filterCriteria conditions.
     *
     * @param FilterCriteria $filterCriteria     The request's filterCriteria.
     * @param Configuration  $dataSourceConfig   Datasource wich the given filterCriteria is related to.
     *
     * @return array List of conditions to be used at the new filterCriteria.
     */
    private function populateParameterBagByDataSourceConfig(FilterCriteria $filterCriteria, Configuration $dataSourceConfig) {
        if ($query = $dataSourceConfig->getQuery()) {
            $params = self::getPlaceholderPositions($query, false);
            $conditions = array();
            foreach ($filterCriteria->getConditions() as $condition) {
                $columnName = $condition['columnName'];
                if (in_array($columnName, $params)) {
                    $this->parameterBag->set($columnName, $condition['value']);
                } else {
                    $conditions[] = $condition;
                }
            }
        } else {
            $conditions = $filterCriteria->getConditions();
        }

        return $conditions;
    }

    private function populateOrderByClause($filterCriteria, $newFilterCriteria){
        foreach ($filterCriteria->getOrderBy() as $columnName => $orderBy) {
            $newFilterCriteria->addOrderBy($columnName, $orderBy);
        }
    }

    /**
     * Builds the filterCriteria.
     *
     * Builds the filterCriteria based on the given one and the actual state of the datasource.
     *
     * @param  FilterCriteria $filterCriteria Filter to base the new filter.
     *
     * @return FilterCriteria Filter for the dataExporter.
     */
    private function populateParameterBag(FilterCriteria $filterCriteria) {
        $dataSourceConfig = $this->dataSourceNameProvider->getDataSourceByName($filterCriteria->getDataSourceName());
        $conditions = $this->populateParameterBagByDataSourceConfig($filterCriteria, $dataSourceConfig);
        $newFilterCriteria =  new FilterCriteria(
            $filterCriteria->getDataSourceName(),
            $conditions,
            $filterCriteria->getPage(),
            $filterCriteria->getPageSize()
        );
        $this->populateOrderByClause($filterCriteria, $newFilterCriteria);
        return $newFilterCriteria;
    }

    /**
     * Creates PDF file and returns it's name.
     *
     * Calls createDataExportFile or generateReportWithData method passing the type PDF to create the file.
     *
     * @param  Request\Filter|Row $request  Header for creating the file.
     * @param  Response           $response Object to set the reponse of the file creation.
     */
    public function createDataExportPDFFile($request, Response $response){
        if($request instanceof Row){
            $this->generateReportWithData($request, $response, DataExporterService::EXPORT_FORMAT_REPORT);
        }
        else {
            $this->createDataExportFile($request, $response, DataExporterService::EXPORT_FORMAT_REPORT);
        }
    }

    /**
     * Creates XLS file and returns it's name.
     *
     * Calls createDataExportFile or generateReportWithData method passing the type XLS to create the file.
     *
     * @param  Request\Filter|Row $request  Header for creating the file.
     * @param  Response       $response Object to set the reponse of the file creation.
     */
    public function createDataExportXLSFile($request, Response $response)
    {
        try {
            if ($request instanceof Row) {
                list($dataSet, $metaData) = $this->getParams($request);
                if (empty($metaData['reportXLS'])) {
                    $this->generateReportWithData($request, $response, DataExporterService::EXPORT_FORMAT_SPREADSHEET);
                } else {
                    if(empty($metaData['formatXLS'] )){
                        $metaData['formatXLS'] = 'F1';
                    }
                    $fileName = $this->spreadsheetWriter->generateXLS($dataSet, $metaData);
                    $message = $this->getMessage($fileName, $metaData);
                    $response->addMessage(new Response\Message($message));
                }

            } else {
                $this->createDataExportFile($request, $response, DataExporterService::EXPORT_FORMAT_SPREADSHEET);
            }
        } catch (\Exception $e) {
            $response->setError(new Response\Error("A unexpected error occurred: {$e->getMessage()}", $e->getCode(), $e->getTraceAsString()));
            throw $e;
        }
    }

    public function generateCSV(Row $request, Response $response) {
        list($dataSet, $metaData) = $this->getParams($request);
        $reportBuilder = $this->csvService;

        try {
            $fileName = $reportBuilder->generateCSV($dataSet, $metaData);
            $message = $this->getMessage($fileName, $metaData);
            $response->addMessage(new Response\Message($message));
        } catch (\Exception $e) {
            $response->setError(new Response\Error($e->getMessage(), $e->getCode()));
        }
    }

    protected function getParams(Row $request){
        $row = $request->getRow();
        $dataSet = json_decode($row["dataSet"], true);
        $metaData = json_decode($row["metaData"], true);
        return array($dataSet, $metaData);
    }

    /**
     * Get the message to return to the frontend.
     *
     * Build the message to return to the fronted, with the id to get the generated report.
     *
     * @param string $fileName - File name of the file.
     * @param array $metadata - Metadata of the generated report.
     * @return string Message to be sent.
     */
    protected function getMessage($fileName, $metadata){
        $metadata['title'] = $this->cleanTitle($metadata['title']);
        $row = array("fileName"=>$fileName, "title"=>$metadata['title']);
        $rowBase = base64_encode(json_encode($row));
        return "/Report?requestType=Row&row[ROW_BASE]=$rowBase";
    }

    protected function cleanTitle($string) {
        $string = str_replace(' ', '-', $string); // Replaces all spaces with hyphens.
        return preg_replace('/[^A-Za-z0-9\-]/', '', $string); // Removes special chars.
    }

    /**
     * Creates the dataExport file.
     *
     * Creates the dataExportFile, by getting the parameters from the given request, and passing to the response the name of the file if the file is created or the error explaining why the file cration goes wrong.
     *
     * @param  Request\Filter $request      Header for creating the file.
     * @param  Response       $response     Object to set the reponse of the file creation.
     * @param  string         $exportFormat The file format to be exported.
     */
    public function createDataExportFile(Request\Filter $request, Response $response, $exportFormat) {
        try {
            $params = $request->getParameters();
            $metadata = $params['metaData'];
            if(isset($params['collectionName'])){
                $metadata['dbType'] = "noSQL";
                $dataSourceName = $params['collectionName'];
            } else {
                $dataSourceName = $params['dataSourceName'];
                $metadata['dbType'] = "SQL";
            }
            $originalFilterCriteria = $request->getFilterCriteria();
            $originalFilterCriteria->setDataSourceName($dataSourceName);
            $filterCriteria = $this->populateParameterBag($originalFilterCriteria);

            $this->startTimeout();

            $fileName = $this->dataExporter->createExportFile($filterCriteria, $metadata, $exportFormat);
            $message = $this->getMessage($fileName, $metadata);
            $response->addMessage(new Response\Message($message));
        } catch (DTOException $e) {
            $response->setError(new Response\Error(
                "Missing data on request: {$e->getMessage()}",
                $e->getCode(),
                $e->getTraceAsString())
            );
        } catch (ServiceException $e) {
            $response->setError(new Response\Error(
                    "A error occurred while trying to generate your file: {$e->getMessage()}",
                    $e->getCode(),
                    $e->getTraceAsString())
            );
        } catch (\Exception $e) {
            $response->setError(new Response\Error("A unexpected error occurred: {$e->getMessage()}", $e->getCode(), $e->getTraceAsString()));
        }
    }

    /**
     * Generate the Report with the data given.
     *
     * Generate the Report, by getting the parameters from the given request, and passing to the response the name of the file if the file is created or the error explaining why the file creation goes wrong.
     *
     * @param  Row            $request      Header for creating the file.
     * @param  Response       $response     Object to set the reponse of the file creation.
     * @param  string         $exportFormat The file format to be exported.
     */
    public function generateReportWithData(Row $request, Response $response, $format) {
        try {
            $row = $request->getRow();
            $metaData = json_decode($row['metaData'], true);
            $dataSet = json_decode($row['dataSet'], true);
            $filter = json_decode($row['filter'], true);

            $orderedDataSet = $this->dataExporter->orderData($filter, $dataSet, $metaData);
            $this->startTimeout();
            $fileName = $this->dataExporter->generateReport($metaData, $orderedDataSet, $format);
            $message = $this->getMessage($fileName, $metaData);

            $response->addMessage(new Response\Message($message));
        } catch (DTOException $e) {
            $response->setError(new Response\Error(
                "Missing data on request: {$e->getMessage()}",
                $e->getCode(),
                $e->getTraceAsString())
            );
        } catch (ServiceException $e) {
            $response->setError(new Response\Error(
                    "A error occurred while trying to generate your file: {$e->getMessage()}",
                    $e->getCode(),
                    $e->getTraceAsString())
            );
        } catch (\Exception $e) {
            $response->setError(new Response\Error("A unexpected error occurred: {$e->getMessage()}", $e->getCode(), $e->getTraceAsString()));
        }
    }

    /**
     *
     * Gives the file extension by it's name.
     *
     * Using the given name as parameter, the method will return the extension on its name.
     *
     * @param string $fileName  The file's name
     *
     * @return string           Returns the file extension by it's name
     */
    private function getFileExtension($fileName){
        return substr($fileName, strrpos($fileName, ".") + 1);
    }

    /**
     *
     * Returns the pdf or xlsx mime types by the given extension  or false if none of them.
     *
     * Using the given extension as parameter, the method will return
     * "application/pdf" or "application/xlsx"  or false if none of them.
     *
     * @param string $extension  The file's extension
     *
     * @throws \Exception File extension not suported.
     *
     * @return string     Returns the pdf or xlsx mime types by the given extension or false if none of them.
     */
    private function getPDFOrXLSXMimeTypeByExtension($extension) {
        if (!isset($this->mimeTypeByExtension[$extension])) {
            throw new \Exception("Unsupported report file extension", 0);
        }

        return $this->mimeTypeByExtension[$extension];
    }

    /**
     * Downloads the generated file.
     *
     * Get the file with the given name, output it with readfile and deletes using unlink.
     *
     * @param  Request  $request  Header for download the file.
     * @param  Response $response Object to set the reponse of the file dowload.
     */
    public function downloadExportedFile(Request $request, Response $response) {
        try {
            $reportName = $request->getParameter('reportName');
            $reportPath = $this->fileNameProvider->getFullFilePath($reportName);
            $mimeType = $this->getPDFOrXLSXMimeTypeByExtension($this->getFileExtension($reportName));

            $file = new Response\File($reportPath, true, $reportName, $mimeType);
            $response->setFileToDownload($file);
        } catch (\Exception $e) {
            $response->setError(new Response\Error($e->getMessage(), $e->getCode(), $e->getTraceAsString(), $e->getPrevious()));
        }
    }
}
