<?php
namespace Zeedhi\DataExporter\Service\ExportStrategy;

use Zeedhi\DataExporter\Service\ExportStrategy;

/**
 * Manages the writing on a escaped spreadsheet file.
 */
class EscapedSpreadSheet implements ExportStrategy {

    /** @var ExportStrategy Used to manipulate the routine of create a file, wich is: start, write rows and finish. */
    private $exportStrategy;

    /**
     * Constructor.
     *
     * @param ExportStrategy $exportStrategy
     */
    public function __construct(ExportStrategy $exportStrategy){
        $this->exportStrategy = $exportStrategy;
    }

    /**
     * Start to generate the file.
     *
     * Calls the function start of the exportStrategy declared on the constructor passing the given metadata.
     *
     * @param  array  $metadata   The definition of the report structure.
     */
    public function start(array $metadata){
        $this->exportStrategy->start($metadata);
    }

    /**
     * Escape the given rows.
     *
     * Escape the value of the given rows.
     *
     * @param  array $rows   Rows to escape.
     *
     * @return array Escaped rows.
     */
    private function getEscapedRows(array $rows){
        foreach($rows as $rowIndex => $row){
            foreach($row as $columnName => $value){
                if(isset($value{0}) && $value{0} == "="){
                    $rows[$rowIndex][$columnName] = "'$value";
                }
            }
        }
        return $rows;
    }

    /**
     * Write the given rows on the file.
     *
     * Escape the given rows and write them using the exportStrategy.
     *
     * @param array  $rows Rows to be written.
     */
    public function writeRows(array $rows){
        $rows = $this->getEscapedRows($rows);
        $this->exportStrategy->writeRows($rows);
    }

    /**
     * Finish the creation of the file.
     *
     * Calls the esportStrategy finish function.
     */
    public function finish(){
        $this->exportStrategy->finish();
    }

    /**
     * Return the file format name.
     *
     * @return string The file format.
     */
    public function formatName(){
        return $this->exportStrategy->formatName();
    }
}