<?php
namespace Zeedhi\DataExporter\Service\DataProvider;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\DriverManager;
use Zeedhi\Framework\DataSource\Configuration;
use Zeedhi\Framework\DataSource\FilterCriteria;
use Zeedhi\Framework\DataSource\ParameterBag;
use Zeedhi\Framework\DataSource\Manager\Doctrine\NameProvider;

class SQLConnectionTest extends \PHPUnit_Framework_TestCase {

    /* @var Connection */
    private $connection;
    /** @var SQLConnection */
    private $dataProvider;
    /* @var ParameterBag|\PHPUnit_Framework_MockObject_MockObject */
    private $parameterBag;
    /* @var NameProvider|\PHPUnit_Framework_MockObject_MockObject */
    private $nameProvider;

    public function setUp() {
        $this->parameterBag = $this->getMockBuilder(ParameterBag::class)
            ->disableOriginalConstructor()
            ->setMethods(array('set', 'get'))
            ->getMock();

        $this->connection = DriverManager::getConnection(array(
            'driver'    => 'oci8',
            'user'      => 'USR_ORG_20',
            'password'  => 'teknisa',
            'host'      => '192.168.122.5',
            'port'      => '1521',
            'dbname'    => 'pdborcl',
            'service'   => true
        ));

        $this->nameProvider = $this->getMockBuilder(NameProvider::class)
            ->getMock();

        $this->dataProvider = new SQLConnection(
            $this->connection,
            $this->nameProvider,
            $this->parameterBag
        );
    }

    public function testSimpleDataSource() {
        $expectedRows = array(
            array('REGION_ID' => 1, 'REGION_NAME' => "Europe"),
            array('REGION_ID' => 2, 'REGION_NAME' => "Americas"),
            array('REGION_ID' => 3, 'REGION_NAME' => "Asia"),
            array('REGION_ID' => 4, 'REGION_NAME' => "Middle East and Africa")
        );
        $filterCriteria = new FilterCriteria("regions");

        $dataSourceConfig = array(
            "tableName" => "REGIONS",
            "columns" => array(
                "REGION_ID",
                "REGION_NAME",
            )
        );

        $this->nameProvider->expects($this->once())
            ->method("getDataSourceByName")
            ->with("regions")
            ->willReturn(EasyConfiguration::factoryFromArrayData("regions", $dataSourceConfig));

        $this->dataProvider->start($filterCriteria);
        $rows = $this->dataProvider->fetch();
        $this->assertEquals($expectedRows, $rows);
        $this->dataProvider->finish();
    }

    public function testSimpleDataSourceWithConditions() {
        $expectedRows = array(
            array('REGION_ID' => 2, 'REGION_NAME' => "Americas"),
            array('REGION_ID' => 3, 'REGION_NAME' => "Asia"),
            array('REGION_ID' => 4, 'REGION_NAME' => "Middle East and Africa")
        );

        $filterCriteria = new FilterCriteria("regions");
        $filterCriteria->addCondition("REGION_NAME", FilterCriteria::LIKE, "%A%");

        $dataSourceConfig = array(
            "tableName" => "REGIONS",
            "columns" => array(
                "REGION_ID",
                "REGION_NAME",
            )
        );

        $this->nameProvider->expects($this->once())
            ->method("getDataSourceByName")
            ->with("regions")
            ->willReturn(EasyConfiguration::factoryFromArrayData("regions", $dataSourceConfig));

        $this->dataProvider->start($filterCriteria);
        $rows = $this->dataProvider->fetch();
        $this->assertEquals($expectedRows, $rows);
        $this->dataProvider->finish();
    }

    public function testDataSourceWithQueries() {
        $expectedRows = array(
            array("COUNTRY_NAME" => "India", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "China", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "Japan", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "Singapore", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "Australia", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "HongKong", "REGION_NAME" => "Asia"),
        );

        $filterCriteria = new FilterCriteria("country_and_region");
        $filterCriteria->addCondition("REGION_NAME", FilterCriteria::LIKE, 'Asia');

        $dataSourceConfig = array(
            "tableName" => "COUNTRY",
            "columns" => array(
                "COUNTRY_NAME",
                "REGION_NAME",
            ),
            "query" => "SELECT COUNTRY_NAME, REGION_NAME FROM COUNTRIES C JOIN REGIONS R ON R.REGION_ID = C.REGION_ID"
        );

        $this->nameProvider->expects($this->once())
            ->method("getDataSourceByName")
            ->with("country_and_region")
            ->willReturn(EasyConfiguration::factoryFromArrayData("country_and_region", $dataSourceConfig));

        $this->dataProvider->start($filterCriteria);
        $rows = $this->dataProvider->fetch();
        $this->assertEquals($expectedRows, $rows);
        $this->dataProvider->finish();
    }

    public function testDataSourceWithParameterBag() {
        $expectedRows = array(
            array("COUNTRY_NAME" => "India", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "China", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "Japan", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "Singapore", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "Australia", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "HongKong", "REGION_NAME" => "Asia"),
        );

        $filterCriteria = new FilterCriteria("country_and_region");
        $filterCriteria->addCondition("REGION_NAME", FilterCriteria::LIKE, 'Asia');

        $dataSourceConfig = array(
            "tableName" => "COUNTRY",
            "columns" => array(
                "COUNTRY_NAME",
                "REGION_NAME",
            ),
            "query" => "SELECT COUNTRY_NAME, REGION_NAME FROM COUNTRIES C JOIN REGIONS R ON R.REGION_ID = C.REGION_ID WHERE R.REGION_NAME LIKE :REGION_NAME"
        );

        $this->parameterBag->expects($this->once())
            ->method("get")
            ->with("REGION_NAME")
            ->willReturn("Asia");

        $this->nameProvider->expects($this->once())
            ->method("getDataSourceByName")
            ->with("country_and_region")
            ->willReturn(EasyConfiguration::factoryFromArrayData("country_and_region", $dataSourceConfig));

        $this->dataProvider->start($filterCriteria);
        $rows = $this->dataProvider->fetch();
        $this->assertEquals($expectedRows, $rows);
        $this->dataProvider->finish();
    }

    public function testDataSourceWithWhereClause() {
        $expectedRows = array(
            array("COUNTRY_NAME" => "India", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "China", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "Japan", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "Singapore", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "Australia", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "HongKong", "REGION_NAME" => "Asia"),
        );

        $filterCriteria = new FilterCriteria("country_and_region");
        $filterCriteria->setWhereClause("REGION_NAME = :REGION_NAME", array(
            "REGION_NAME" => "Asia"
        ));

        $dataSourceConfig = array(
            "tableName" => "COUNTRIES",
            "columns" => array(
                "COUNTRY_NAME",
                "REGION_NAME"
            ),
            "query" => "SELECT COUNTRY_NAME, REGION_NAME FROM COUNTRIES C JOIN REGIONS R ON R.REGION_ID = C.REGION_ID"
        );

        $this->nameProvider->expects($this->once())
            ->method("getDataSourceByName")
            ->with("country_and_region")
            ->willReturn(EasyConfiguration::factoryFromArrayData("country_and_region", $dataSourceConfig));

        $this->dataProvider->start($filterCriteria);
        $rows = $this->dataProvider->fetch();
        $this->assertEquals($expectedRows, $rows);
        $this->dataProvider->finish();
    }

    public function testDataSourceWithDataSourceConditions() {
        $expectedRows = array(
            array("COUNTRY_NAME" => "India", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "China", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "Japan", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "Singapore", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "Australia", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "HongKong", "REGION_NAME" => "Asia"),
        );

        $filterCriteria = new FilterCriteria("asia_country_and_region");

        $dataSourceConfig = array(
            "tableName" => "COUNTRIES",
            "columns" => array(
                "COUNTRY_NAME",
                "REGION_NAME"
            ),
            "query" => "SELECT COUNTRY_NAME, REGION_NAME FROM COUNTRIES C JOIN REGIONS R ON R.REGION_ID = C.REGION_ID",
            "conditions" => array(
                "REGION_NAME = 'Asia'"
            )
        );

        $this->nameProvider
            ->expects($this->once())
            ->method("getDataSourceByName")
            ->with("asia_country_and_region")
            ->willReturn(EasyConfiguration::factoryFromArrayData("asia_country_and_region", $dataSourceConfig));

        $this->dataProvider->start($filterCriteria);
        $rows = $this->dataProvider->fetch();
        $this->assertEquals($expectedRows, $rows);
        $this->dataProvider->finish();
    }

    public function testDataSourceWithOrderBy() {
        $expectedRows = array(
            array("COUNTRY_NAME" => "Singapore", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "Japan", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "India", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "HongKong", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "China", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "Australia", "REGION_NAME" => "Asia"),
        );

        $filterCriteria = new FilterCriteria("country_and_region");
        $filterCriteria->addOrderBy("COUNTRY_NAME", FilterCriteria::ORDER_DESC);
        $filterCriteria->setWhereClause("REGION_NAME = :REGION_NAME", array(
            "REGION_NAME" => "Asia"
        ));

        $dataSourceConfig = array(
            "tableName" => "COUNTRIES",
            "columns" => array(
                "COUNTRY_NAME",
                "REGION_NAME"
            ),
            "query" => "SELECT COUNTRY_NAME, REGION_NAME FROM COUNTRIES C JOIN REGIONS R ON R.REGION_ID = C.REGION_ID"
        );

        $this->nameProvider->expects($this->once())
            ->method("getDataSourceByName")
            ->with("country_and_region")
            ->willReturn(EasyConfiguration::factoryFromArrayData("country_and_region", $dataSourceConfig));

        $this->dataProvider->start($filterCriteria);
        $rows = $this->dataProvider->fetch();
        $this->assertEquals($expectedRows, $rows);
        $this->dataProvider->finish();
    }

    public function testDataSourceWithPaginatedResults() {
        $expectedRows = array(
            array("COUNTRY_NAME" => "Australia", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "China", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "HongKong", "REGION_NAME" => "Asia"),
        );

        $filterCriteria = new FilterCriteria("asia_country_and_region");
        $filterCriteria->setPage(1);
        $filterCriteria->setPageSize(3);

        $dataSourceConfig = array(
            "tableName" => "COUNTRIES",
            "columns" => array(
                "COUNTRY_NAME",
                "REGION_NAME"
            ),
            "query" => "SELECT COUNTRY_NAME, REGION_NAME FROM COUNTRIES C JOIN REGIONS R ON R.REGION_ID = C.REGION_ID",
            "conditions" => array(
                "REGION_NAME = 'Asia'"
            )
        );

        $this->nameProvider
            ->expects($this->once())
            ->method("getDataSourceByName")
            ->with("asia_country_and_region")
            ->willReturn(EasyConfiguration::factoryFromArrayData("asia_country_and_region", $dataSourceConfig));

        $this->dataProvider->start($filterCriteria);
        $rows = $this->dataProvider->fetch();
        $this->assertEquals($expectedRows, $rows);
        $this->dataProvider->finish();
    }

    public function testDataSourceResultSetLimit() {
        $expectedRows = array(
            array("COUNTRY_NAME" => "Australia", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "China", "REGION_NAME" => "Asia"),
            array("COUNTRY_NAME" => "HongKong", "REGION_NAME" => "Asia"),
        );

        $filterCriteria = new FilterCriteria("asia_country_and_region");

        $dataSourceConfig = array(
            "tableName" => "COUNTRIES",
            "columns" => array(
                "COUNTRY_NAME",
                "REGION_NAME"
            ),
            "query" => "SELECT COUNTRY_NAME, REGION_NAME FROM COUNTRIES C JOIN REGIONS R ON R.REGION_ID = C.REGION_ID",
            "resultSetLimit" => 3,
            "conditions" => array(
                "REGION_NAME = 'Asia'"
            )
        );

        $this->nameProvider
            ->expects($this->once())
            ->method("getDataSourceByName")
            ->with("asia_country_and_region")
            ->willReturn(EasyConfiguration::factoryFromArrayData("asia_country_and_region", $dataSourceConfig));

        $this->dataProvider->start($filterCriteria);
        $rows = $this->dataProvider->fetch();
        $this->assertEquals($expectedRows, $rows);
        $this->dataProvider->finish();
    }

    public function testBuildStatementWithInvalidFieldName() {
        $this->expectException(\Zeedhi\Framework\DataSource\Manager\Exception::class);
        $this->expectExceptionMessage('Column INVALID_FIELD was not found in result set of data source country_and_region.');

        $filterCriteria = new FilterCriteria("country_and_region");
        $filterCriteria->addCondition("REGION_NAME", FilterCriteria::LIKE, 'Asia');

        $dataSourceConfig = array(
            "tableName" => "COUNTRY",
            "columns" => array(
                "COUNTRY_NAME",
                "REGION_NAME",
                "INVALID_FIELD"
            ),
            "query" => "SELECT COUNTRY_NAME, REGION_NAME FROM COUNTRIES C JOIN REGIONS R ON R.REGION_ID = C.REGION_ID WHERE R.REGION_NAME LIKE :REGION_NAME"
        );

        $this->parameterBag
            ->expects($this->once())
            ->method("get")
            ->with("REGION_NAME")
            ->willReturn("Asia");

        $this->nameProvider->expects($this->once())
            ->method("getDataSourceByName")
            ->with("country_and_region")
            ->willReturn(EasyConfiguration::factoryFromArrayData("country_and_region", $dataSourceConfig));

        $this->dataProvider->start($filterCriteria);
    }

}

class EasyConfiguration extends Configuration {
    //sorry about this, i'll put in another file later.
    public static function factoryFromArrayData($dataSourceName, $dataSourceConfigData) {
        return static::factoryFromJsonData($dataSourceConfigData, $dataSourceName);
    }
}