<?php
namespace Zeedhi\FileServerAPI;

use Zeedhi\Framework\DTO\Response;
use Zeedhi\Framework\DataSource\DataSet;
use Zeedhi\Framework\Remote\RequestFactory;
use Zeedhi\Framework\Remote\Server;
use Zeedhi\Framework\Remote\Exception as RemoteException;

/**
 * File Server API contract implementation.
 */
class API {

    /** @var string The APIKey header name */
    const API_KEY = 'apiKey';

    /** @var Server Connection with remote file server */
    protected $fileServer;
    /** @var RequestFactory Helper that creates requests */
    protected $requestFactory;

    /**
     * @param Server         $fileServer
     * @param RequestFactory $requestFactory
     */
    public function __construct(Server $fileServer, RequestFactory $requestFactory) {
        $this->fileServer = $fileServer;
        $this->requestFactory = $requestFactory;
    }

    /**
     * Dumps the organizations list.
     * @return array All organizations objects.
     */
    public function dumpOrganizations() {
        $request = $this->requestFactory->createEmptyRequest('GET', '/organizations');
        $response = $this->fileServer->request($request);

        $this->checkResponseError($response);

        return $response->getDataSets();
    }

    /**
     * Updates an organization.
     * @param string $organizationName Name of the organization
     * @param array $newOrganizationObject The up to date organization object.
     * @return array The up to date object.
     */
    public function updateOrganization($organizationName, $newOrganizationObject) {
        $row = [
            'organizationName' => $organizationName, 
            'organizationObject' => $newOrganizationObject
        ];

        $request = $this->requestFactory->createRowRequest('POST', '/organization/update', $row);
        $response = $this->fileServer->request($request);

        $this->checkResponseError($response);

        return $response->getDataSets();
    }

    /**
     * Created an organization.
     * @param array $organizationObject The new organization object.
     * @return array The new organization object.
     */
    public function createOrganization($organizationObject) {
        $request = $this->requestFactory->createRowRequest('POST', '/organization/create', $organizationObject);
        $response = $this->fileServer->request($request);

        $this->checkResponseError($response);

        return $response->getDataSets();
    }

    /**
     * Deletes an organization.
     * @param string $organizationName Name of the organization
     * @return array The number of deleted organizations.
     */
    public function deleteOrganization($organizationName) {
        $request = $this->requestFactory->createRowRequest('POST', '/organization/delete', ['name' => $organizationName]);
        $response = $this->fileServer->request($request);

        $this->checkResponseError($response);

        return $response->getDataSets();
    }

    /**
     * Dumps the files list.
     * @return array All files objects.
     */
    public function dumpFiles() {
        $request = $this->requestFactory->createEmptyRequest('GET', '/dump');
        $response = $this->fileServer->request($request);

        $this->checkResponseError($response);

        return $response->getDataSets();
    }

    /**
     * Uploads a single file.
     * @param string $file The file to upload.
     * @return array The persisted file.
     */
    public function uploadFile($file) {
        $request = $this->requestFactory->createRowRequest('POST', '/upload', $file);
        $response = $this->fileServer->request($request);

        $this->checkResponseError($response);

        return $response->getDataSets();
    }

    /**
     * Uploads multiple files.
     * @param array $files The files to persist.
     * @return array The persisted files.
     */
    public function uploadFiles($files) {
        $request = $this->requestFactory->createDataSetRequest('POST', '/upload', new Dataset('files', $files));
        $response = $this->fileServer->request($request);

        $this->checkResponseError($response);

        return $response->getDataSets();
    }

    /**
     * Deletes a file.
     * @param string $filePath The file path.
     * @return array The number of deleted files.
     */
    public function deleteFile($filePath) {
        $request = $this->requestFactory->createRowRequest('POST', '/delete', ['PATH' => $filePath]);
        $response = $this->fileServer->request($request);

        $this->checkResponseError($response);

        return $response->getDataSets();
    }

    /**
     * Retrieves file content.
     * @param string $filePath The file path.
     * @return array The file content.
     */
    public function downloadFile($filePath) {
        $request = $this->requestFactory->createEmptyRequest('GET', '/files' . $filePath);
        return $this->fileServer->fileRequest($request);
    }

    /**
     * Checks the response content for errors.
     * @param Response $response The Zeedhi DTO Response Object.
     * @throws RemoteException Thrown if an error is encountered in the response.
     */
    protected function checkResponseError(Response $response) {
        if ($response->getError()) {
            throw RemoteException::remoteServerError($response->getError());
        }
    }

}
