<?php
namespace Zeedhi\Framework\DataSource\Manager\IdProvider;

use Zeedhi\Framework\DataSource\Configuration;
use Zeedhi\Framework\DataSource\Manager;
use Zeedhi\Framework\DataSource\FilterCriteria;
use Zeedhi\Framework\DataSource\DataSet;


class ManagerImpl implements Manager {

    /** @var Manager */
    protected $manager;
    /** @var \Zeedhi\Framework\DataSource\Manager\IdProvider\IdProvider */
    protected $idProvider;
    /** @var Manager\Doctrine\NameProvider */
    protected $nameProvider;
    /** @var string */
    protected $dataSourceName;
    /** @var Configuration */
    protected $dataSourceConfig;

    public const ALL_DATA = "__ALL";

    public function __construct(Manager\Doctrine\NameProvider $nameProvider, Manager $manager, IdProvider $idProvider){
        $this->manager = $manager;
        $this->idProvider = $idProvider;
        $this->nameProvider = $nameProvider;
    }

    /**
     * Persist all given rows in DataSet.
     *
     * @param DataSet $dataSet The collection and description of rows.
     *
     * @return array Rows with primary key columns values.
     *
     * @throws Exception
     */
    public function persist(DataSet $dataSet) {
        $this->dataSourceName = $dataSet->getDataSourceName();
        $this->dataSourceConfig = $this->nameProvider->getDataSourceByName($this->dataSourceName);
        $tableName = $this->dataSourceConfig->getTableName();
        $columnName = $this->dataSourceConfig->getSequentialColumn();
        foreach($dataSet->getRows() as $row) {
            if ($row['__is_new']) {
                $row[$columnName] = $this->idProvider->getNextId($tableName, $columnName);
            }
        }
        return $this->manager->persist($dataSet);
    }

    /**
     * Persist all given rows in DataSet on Next structure.
     *
     * @param DataSet $dataSet The collection and description of rows.
     * @param bool $isNew True if is a new DataSet false if not.
     *
     * @return array Rows with primary key columns values.
     *
     * @throws Exception
     */
    public function persistForNext(DataSet $dataSet, bool $isNew) {
        $this->dataSourceName = $dataSet->getDataSourceName();
        $this->dataSourceConfig = $this->nameProvider->getDataSourceByName($this->dataSourceName);
        $tableName = $this->dataSourceConfig->getTableName();
        $columnName = $this->dataSourceConfig->getSequentialColumn();
        if($isNew && !isset($dataSet->getRows()[0][$columnName]) && !is_null($columnName)) {
            $newRows = $dataSet->getRows();
            foreach($newRows as $key=>$row) {
                $newRows[$key][$columnName] = $this->idProvider->getNextId($tableName, $columnName);
            }
            $dataSet->setRows($newRows);
        }
        return $this->manager->persistForNext($dataSet, $isNew);
    }

    /**
     * Delete all given rows in DataSet.
     *
     * @param DataSet $dataSet The collection and description of rows.
     *
     * @return array Rows with primary key columns values.
     *
     * @throws Exception
     */
    public function delete(DataSet $dataSet) {
        return $this->manager->delete($dataSet);
    }

    /**
     * Delete all given rows in DataSet on Next structure.
     *
     * @param DataSet $dataSet The collection and description of rows.
     *
     * @return array Rows with primary key columns values.
     *
     * @throws Exception
     */
    public function deleteForNext(DataSet $dataSet) : array {
        return $this->manager->deleteForNext($dataSet);
    }

    /**
     * Return a DataSet with rows that match the given criteria.
     *
     * @param FilterCriteria $filterCriteria
     *
     * @return mixed The result of the filter criteria.
     */
    public function findBy(FilterCriteria $filterCriteria) {
        return $this->manager->findBy($filterCriteria);
    }

    /**
     * Return a DataSet with rows and pagination that match the given criteria.
     *
     * @param FilterCriteria $filterCriteria
     *
     * @return DataSet The result of the filter criteria.
     */
    public function findByForNext(FilterCriteria $filterCriteria) : DataSet {
        return $this->manager->findByForNext($filterCriteria);
    }

    /**
    * Verify if the dataSet contains a ALL_DATA's flag and populate it.
    *
    * @param DataSet $dataSet
    *
    * @return DataSet $dataSet A populated dataSet.
    */
    public function populateDataSet(DataSet $dataSet) {
        $rows = $dataSet->getRows();
        $dataSourceName = $dataSet->getDataSourceName();
        foreach($rows[0] as $column => $value) {
            if(is_array($value) && isset($value[self::ALL_DATA])) {
                if(isset($data)) {
                    $rows[0][$column] = $data;
                } else {
                    $filterCriteria = $this->buildAllDataFilter($rows, $value, $column, $dataSourceName);
                    $data = $rows[0][$column] = $this->findBy($filterCriteria)->getRows();
                }
            }
        }
        return new DataSet($dataSourceName, $rows);
    }

    /**
    * Build a filterCriteria based on dataSourceFilter.
    *
    * @param $rows
    * @param $value
    * @param $column
    *
    * @return FilterCriteria $filterCriteria
    */
    public function buildAllDataFilter($rows, $value, $column, $dataSourceName) {
        $filterCriteria = new FilterCriteria($dataSourceName);
        if($rows[0][$column . '_EXCEPT'] ?? false) {
            $exceptionFilter = [];
            foreach($rows[0][$column . '_EXCEPT'] as $exceptRow) {
                $exceptionFilter[] = $exceptRow;
            }
            $filterCriteria->addCondition($column, "NOT_IN", $exceptionFilter);
        }
        foreach($value[self::ALL_DATA] as $filter) {
            $filterCriteria->addCondition($filter["name"], $filter["operator"], $filter["value"]);
        }
        return $filterCriteria;
    }

    /**
     * @see Zeedhi\Framework\DataSource\Manager\AbstractManager::isViewRequest
     */
    public function isViewRequest(array $parameters) : bool {
        return $this->manager->isViewRequest($parameters);
    }

    /**
     * @see Zeedhi\Framework\DataSource\Manager\AbstractManager::getQueryParametersFromJson
     */
    public function getQueryParametersFromJson(FilterCriteria $filterCriteria) : array {
        return $this->manager->getQueryParametersFromJson($filterCriteria);
    }
}