<?php
namespace Zeedhi\Framework\Routing\Next;

use Zeedhi\Framework\DataSource\Manager\Doctrine\NameProvider;
use Zeedhi\Framework\Service\Next;
use Zeedhi\Framework\Routing\Route;
use Zeedhi\Framework\Routing\Router;
use Zeedhi\Framework\DTO;
use Zeedhi\Framework\DataSource\Configuration;

class RouteFactory {

    public const CONTROLLER_NAME = 'crudNextController';
    public const REQUEST_METHOD_MAPPING = [
        Router::METHOD_GET => 'find',
        Router::METHOD_POST => 'persist',
        Router::METHOD_PUT => 'persist',
        Router::METHOD_DELETE => 'delete'
    ];

    /** @var NameProvider */
    private $nameProvider;
    /** @var Next */
    private $next;

    /**
     * Constructor
     */
    public function __construct(NameProvider $nameProvider) {
        $this->nameProvider = $nameProvider;
        $this->next = new Next($nameProvider);
    }

    /**
     * Build a new route for the request if not exists and add on Router.
     *
     * @param string $uri
     * @param string $method
     * @param string $controllerMethod Optional
     * @param string $controllerId     Optional
     * @param string $dataSourcename   Optional
     */
    public function buildRoute(string $uri, string $method, $controllerMethod = null, $controllerId = self::CONTROLLER_NAME) {
        $arrayUri = Next::buildArrayUri($uri);
        $dataSourceName = $this->next->getDataSourceName($arrayUri);
        $dataSource = $this->nameProvider->getDataSourceByName($dataSourceName);
        $routePath = $this->buildRoutePath($arrayUri, $dataSource);
        $route = Router::getRouteByPath($routePath, $method);
        if(!$route) {
            $parameters = $this->buildParameters($arrayUri, $dataSource);
            $controllerMethod = $controllerMethod ?: $this->getControllerMethod($method);
            $route = new Route(
                [ $method ],
                $routePath,
                $controllerId,
                $controllerMethod,
                DTO\Request::TYPE_NEXT,
                $parameters
            );
            Router::addRoute($route);
        }
    }

    /**
     * Builds a route path with the name of datasources and primary keys.
     *
     * @param array  $arrayUri  The array with the elements of uri (without query).
     * @param string $routePath Route path to be appended.
     *
     * @return string
     */
    private function buildRoutePath(array $arrayUri, Configuration $dataSourceChild, string $routePath = "") : string {
        $dataSourceName = array_shift($arrayUri);
        $routePath .= $dataSourceName ? '/'.$dataSourceName : '';
        if($arrayUri) {
            $dataSource = $this->nameProvider->getDataSourceByName($dataSourceName);
            $primaryKeys = $dataSource->getPrimaryKeyColumns();
            foreach($primaryKeys as $column) {
                $column = $this->getLocalColumnFromRelation($dataSourceChild, $dataSource->getTableName(), $column);
                $routePath .= '/{'.$column.'}';
                array_shift($arrayUri);
            }
            $routePath = $this->buildRoutePath($arrayUri, $dataSourceChild, $routePath);
        }
        return $routePath;
    }

    /**
     * Builds an array with parameters of route.
     *
     * @param array         $arrayUri        The array with the elements of uri (without query).
     * @param Configuration $dataSourceChild The data source child.
     * @param array         $parameters      Parameters to be appended.
     *
     * @return array
     */
    private function buildParameters(array $arrayUri, Configuration $dataSourceChild, array $parameters = []) : array {
        $dataSourceName = array_shift($arrayUri);
        if($arrayUri) {
            $currentDataSource = $this->nameProvider->getDataSourceByName($dataSourceName);
            $primaryKeys = $currentDataSource->getPrimaryKeyColumns();
            foreach($primaryKeys as $column) {
                $column = $this->getLocalColumnFromRelation($dataSourceChild, $currentDataSource->getTableName(), $column);
                $parameters[] = [
                    "name" => $column,
                    "regex" => ".+"
                ];
                array_shift($arrayUri);
            }
            $parameters = $this->buildParameters($arrayUri, $dataSourceChild, $parameters);
        }
        return $parameters;
    }

    /**
     * Get local column name.
     *
     * @param Configuration $dataSource
     * @param string $targetTable
     * @param string $targetColumn
     */
    private function getLocalColumnFromRelation(Configuration $dataSource, string $targetTable, string $targetColumn) {
        if($targetTable != $dataSource->getTableName()) {
            $relations = $dataSource->getRelations();
            foreach($relations as $relation) {
                if($targetTable == $relation['targetTable']) {
                    $localColumnKey = array_search($targetColumn, $relation['targetColumns']);
                    return $relation['localColumns'][$localColumnKey];
                }
            }
        }
        return $targetColumn;
    }

    /**
     * Gets the controller method with rules of Next Server Communications.
     *
     * @param string $method The requested method.
     *
     * @return string Controller method.
     */
    private function getControllerMethod(string $method) : string {
        return self::REQUEST_METHOD_MAPPING[$method];
    }
}