<?php
namespace Zeedhi\Framework\DataSource\Manager;

use Doctrine\DBAL\Query\QueryBuilder;
use Doctrine\ORM\Query\ParameterTypeInferer;
use Zeedhi\Framework\DataSource\AssociatedWithDataSource;
use Zeedhi\Framework\DataSource\Configuration;
use Zeedhi\Framework\DataSource\DataSet;
use Zeedhi\Framework\DataSource\FilterCriteria;
use Zeedhi\Framework\DataSource\Manager;
use Zeedhi\Framework\DataSource\Manager\Doctrine\NameProvider;
use Zeedhi\Framework\DataSource\Operator\DefaultOperator;
use Zeedhi\Framework\DataSource\ParameterBag;
use Zeedhi\Framework\HTTP\Request;

abstract class AbstractManager implements Manager{

    /** @var NameProvider */
    protected $nameProvider;
    /** @var ParameterBag */
    protected $parameterBag;
    /** @var Configuration */
    protected $dataSourceConfig;
    protected $httpRequest;

    const ALL_DATA = "__ALL";

    public function __construct(NameProvider $nameProvider, ParameterBag $parameterBag) {
        $this->nameProvider = $nameProvider;
        $this->parameterBag = $parameterBag;
    }

    /**
	 * Return a populated dataSet.
	 *
	 * Verify if the dataSet contains a ALL_DATA's flag and populate it.
	 *
	 * @param DataSet $dataSet
	 *
	 * @return DataSet $dataSet
	 */
	public function populateDataSet(DataSet $dataSet){
        $rows = $dataSet->getRows();
        $dataSourceName = $dataSet->getDataSourceName();
		foreach($rows[0] as $column => $value){
            if(is_array($value) && isset($value[self::ALL_DATA])){
                if(isset($data)){
                    $rows[0][$column] = $data;
                } else {
                    $filterCriteria = $this->buildAllDataFilter($rows, $value, $column, $dataSourceName);
                    $data = $rows[0][$column] = $this->findBy($filterCriteria)->getRows();
                }
            }
		}
		return new DataSet($dataSourceName, $rows);
    }

    /**
     * Return a filterCriteria.
     *
     * Build a filterCriteria based on dataSourceFilter.
     *
     * @param $rows
     * @param $value
     * @param $column
     *
     * @return FilterCriteria $filterCriteria
     */
    public function buildAllDataFilter($rows, $value, $column, $dataSourceName){
        $filterCriteria = new FilterCriteria($dataSourceName);
        if(isset($rows[0][$column . '_EXCEPT']) && !empty($rows[0][$column . '_EXCEPT'])){
            $exceptionFilter = array();
            foreach($rows[0][$column . '_EXCEPT'] as $exceptRow){
                $exceptionFilter[] = $exceptRow;
            }
            $filterCriteria->addCondition($column, "NOT_IN", $exceptionFilter);
        }
        if(!empty($value[self::ALL_DATA])){
            foreach($value[self::ALL_DATA] as $filter){
                $filterCriteria->addCondition($filter["name"], $filter["operator"], $filter["value"]);
            }
        }
        return $filterCriteria;
    }

    /**
     * @param AssociatedWithDataSource $associatedWithDataSource
     */
    protected function loadCurrentDataSource(AssociatedWithDataSource $associatedWithDataSource) {
        $this->dataSourceConfig = $this->nameProvider->getDataSourceByName($associatedWithDataSource->getDataSourceName());
        $this->dataSourceConfig->setDriverName($this->driverName);
    }

    /**
     * @param $row
     * @return array
     */
    protected function getPrimaryKeyValueFromRow($row) {
        $persistedRow = array();
        foreach ($this->dataSourceConfig->getPrimaryKeyColumns() as $columnName) {
            $persistedRow[$columnName] = $row[$columnName];
        }

        return $persistedRow;
    }


    abstract protected function beginTransaction();
    abstract protected function commit();
    abstract protected function rollback();
    abstract protected function persistRow(array $row);
    abstract protected function persistRowForNext(array $row, bool $isNew) : array;

    protected function persistRows(DataSet $dataSet) {
        $persistedRows = array();
        foreach ($dataSet->getRows() as $key => $row) {
            $this->persistRow($row);
            $persistedRows[$key] = $this->getPrimaryKeyValueFromRow($row);
        }

        return $persistedRows;
    }

    /**
     * Persist all given rows in DataSet.
     *
     * @param DataSet $dataSet The collection and description of rows.
     *
     * @return array Rows with primary key columns values.
     *
     * @throws
     */
    public function persist(DataSet $dataSet) {
        $this->loadCurrentDataSource($dataSet);
        $this->beginTransaction();
        try {
            $persistedRows = $this->persistRows($dataSet);
            $this->commit();
            return $persistedRows;
        } catch (\Exception $e) {
            $this->rollback();
            throw Exception::errorExecutingQuery($e);
        }
    }

    /**
     * Persist all given rows in DataSet on Next structure.
     *
     * @param DataSet $dataSet The collection and description of rows.
     * @param bool $isNew True if is a new DataSet false if not.
     *
     * @return array Rows with primary key columns values.
     *
     * @throws Exception
     */
    public function persistForNext(DataSet $dataSet, bool $isNew) : array {
        $this->loadCurrentDataSource($dataSet);
        $this->beginTransaction();
        try {
            $rows = $dataSet->getRows();
            $row = array_shift($rows);
            $persistedRow = $this->persistRowForNext($row, $isNew);
            $this->commit();
            return $persistedRow;
        } catch (\Exception $e) {
            $this->rollback();
            throw Exception::errorExecutingQuery($e);
        }
    }

    abstract protected function deleteRow(array $row);

    /**
     * @param DataSet $dataSet
     * @return array
     */
    protected function deleteRows(DataSet $dataSet) {
        $deletedRows = array();
        foreach ($dataSet->getRows() as $key => $row) {
            // New rows doesn't need to be deleted, since their do not exist.
            if ($row['__is_new'] === false) {
                $this->deleteRow($row);
                $deletedRows[$key] = $this->getPrimaryKeyValueFromRow($row);
            }
        }

        return $deletedRows;
    }

    /**
     * Delete all given rows in DataSet.
     *
     * @param DataSet $dataSet The collection and description of rows.
     *
     * @return array Rows with primary key columns values.
     *
     * @throws Exception
     */
    public function delete(DataSet $dataSet) {
        $this->loadCurrentDataSource($dataSet);
        $this->beginTransaction();
        try {
            $deletedRows = $this->deleteRows($dataSet);
            $this->commit();
            return $deletedRows;
        } catch (\Exception $e) {
            $this->rollback();
            throw Exception::errorExecutingQuery($e);
        }
    }

    /**
     * Delete all given rows in DataSet on Next structure.
     *
     * @param DataSet $dataSet The collection and description of rows.
     *
     * @return array Rows with primary key columns values.
     *
     * @throws Exception
     */
    public function deleteForNext(DataSet $dataSet) {
        $this->loadCurrentDataSource($dataSet);
        $this->beginTransaction();
        $rows = $dataSet->getRows();
        $row = array_shift($rows);
        try {
            $this->deleteRow($row);
            $this->commit();
            return $this->getPrimaryKeyValueFromRow($row);
        } catch (\Exception $e) {
            $this->rollback();
            throw Exception::errorExecutingQuery($e);
        }
    }

    /**
     * @param FilterCriteria $filterCriteria
     *
     * @return array[] The rows
     */
    abstract protected function retrieveRows(FilterCriteria $filterCriteria);

    /**
     * @param array $rows
     *
     * @return array
     */
    protected function addIsNewColumn(array $rows) : array {
        foreach ($rows as $index => $row) {
            $rows[$index]['__is_new'] = false;
        }
        return $rows;
    }

    /**
     * @param FilterCriteria $filterCriteria
     * @param QueryBuilder $query
     */
    protected function processPagination(FilterCriteria $filterCriteria, QueryBuilder $query) {
        if ($filterCriteria->isPaginated()) {
            $query->setFirstResult($filterCriteria->getFirstResult());
            $query->setMaxResults($filterCriteria->getPageSize());
        } else {
            if ($maxResults = $this->dataSourceConfig->getResultSetLimit()) {
                $query->setMaxResults($maxResults);
            }
        }
    }

    /**
     * @param QueryBuilder $query
     * @param $orderBy
     */
    protected function addOrderByToQuery(QueryBuilder $query, $orderBy) {
        foreach ($orderBy as $columnName => $direction) {
            $query->addOrderBy($columnName, $direction);
        }
    }

    /**
     * @param FilterCriteria $filterCriteria
     * @param QueryBuilder $query
     */
    protected function processOrderBy(FilterCriteria $filterCriteria, QueryBuilder $query) {
        $this->addOrderByToQuery($query, $this->dataSourceConfig->getOrderBy());
        $this->addOrderByToQuery($query, $filterCriteria->getOrderBy());
    }

    /**
     * @param QueryBuilder $query
     * @param $groupBy
     */
    protected function addGroupByToQuery(QueryBuilder $query, $groupBy) {
        $query->addGroupBy($groupBy);
    }

    /**
     * @param FilterCriteria $filterCriteria
     * @param QueryBuilder $query
     */
    protected function processGroupBy(FilterCriteria $filterCriteria, QueryBuilder $query) {
        $this->addGroupByToQuery($query, $this->dataSourceConfig->getGroupBy());
        $this->addGroupByToQuery($query, $filterCriteria->getGroupBy());
    }
    
    /**
     * Build query conditions using zeedhi next dataSource routine.
     * 
     * @param array $conditions
     * @param QueryBuilder $query
     * 
     * @return array
     */
    protected function buildQueryConditionsForNext(array $conditions, QueryBuilder $query) : array {
        $parameters = $this->dataSourceConfig->getParameters();
        $defaults = $this->dataSourceConfig->getDefaults();
        $paramsMap = [];
    
        if($conditions){
            foreach($conditions as $condition) {
                $paramName = $condition['columnName'];
                if(isset($parameters[$paramName]) && $condition['operator'] === FilterCriteria::EQ) {
                    $paramsMap[$paramName] = $this->resolveParameterValue($paramName, $condition['value'] ?? $defaults[$paramName]);
                    unset($parameters[$paramName]);
                } else {
                    if(!isset($condition['value'])){
                        DefaultOperator::factoryFromStringRepresentation($condition['operator'], $this->dataSourceConfig)
                            ->addFunctionToQuery($condition, $query, $paramsMap);
                    }else{
                        DefaultOperator::factoryFromStringRepresentation($condition['operator'], $this->dataSourceConfig)
                            ->addConditionToQuery($condition, $query, $paramsMap);
                    }
                }
            }
        }

        if($parameters) {
            foreach($parameters as $paramName) {
                $paramsMap[$paramName] = $this->resolveParameterValue($paramName, $defaults[$paramName] ?? null);
                unset($parameters[$paramName]);
            }

        }

        return $paramsMap;
    }

    /**
     * Build query conditions using old zeedhi dataSource routine.
     * 
     * @param array $conditions
     * @param QueryBuilder $query
     * 
     * @return array
     */
    protected function buildQueryConditions(array $conditions, QueryBuilder $query) : array {
        $params = [];
    
        foreach($conditions as $condition) {
            DefaultOperator::factoryFromStringRepresentation($condition['operator'], $this->dataSourceConfig)
                ->addConditionToQuery($condition, $query, $params);
        }

        return $params;
    }

    /**
     * Retrieves the value for the parameter from the parameter bag or json defaults.
     * @param string $paramName The parameter name to search for.
     * @param mixed  $default   Default value to use if no value is found.
     * @return mixed The value.
     */
    protected function resolveParameterValue(string $paramName, $default) {
        return $this->parameterBag->getOrDefault($paramName, $default);
    }

    /**
     * @param QueryBuilder $query
     * @param array $params
     * @return array
     */
    protected function completeParamsWithParameterBag(QueryBuilder $query, array $params) : array {
        $matches = [];
        preg_match_all('/(?<=:)\w+/', $query->getSQL(), $matches);
        foreach (reset($matches) as $paramName) {
            $params[$paramName] = $params[$paramName] ?? $this->parameterBag->get($paramName);
        }
        return $params;
    }

    /**
     * Process where clauses using old zeedhi dataSource routine.
     * 
     * @param FilterCriteria $filterCriteria
     * @param QueryBuilder $query
     *
     * @return array
     */
    protected function processWhereClause(FilterCriteria $filterCriteria, QueryBuilder $query) : array {
        $params = $this->buildQueryConditions($filterCriteria->getConditions(), $query);
        
        if($filterCriteria->hasWhereClause()) {
            $query->andWhere($filterCriteria->getWhereClause());
            foreach ($filterCriteria->getWhereClauseParams() as $bindColumnName => $value) {
                $bindColumnName = ltrim($bindColumnName, ':');
                $params[$bindColumnName] = $value;
            }
        }

        foreach ($this->dataSourceConfig->getConditions() as $condition) {
            $query->andWhere($condition);
        }

        return $this->completeParamsWithParameterBag($query, $params);
    }


    /**
     * Process where clauses using zeedhi next dataSource routine.
     * 
     * @param FilterCriteria $filterCriteria
     * @param QueryBuilder $query
     *
     * @return array
     */
    protected function processWhereClauseForNext(FilterCriteria $filterCriteria, QueryBuilder $query) : array {
        $params = $this->buildQueryConditionsForNext($filterCriteria->getConditions(), $query);
        
        if($filterCriteria->hasWhereClause()) {
            $query->andWhere($filterCriteria->getWhereClause());
            foreach ($filterCriteria->getWhereClauseParams() as $bindColumnName => $value) {
                $bindColumnName = ltrim($bindColumnName, ':');
                $params[$bindColumnName] = $value;
            }
        }

        foreach ($this->dataSourceConfig->getConditions() as $condition) {
            $query->andWhere($condition);
        }

        return $this->completeParamsWithParameterBag($query, $params);
    }

    /**
     * Apply FilterCriteria conditions, whereClause and pagination to query.
     * Return a array of parameters needed to execute the query.
     *
     * @param FilterCriteria $filterCriteria
     * @param QueryBuilder   $query
     *
     * @return array
     */
    protected function processFilterConditions(FilterCriteria $filterCriteria, QueryBuilder $query) {
        $params = $this->processWhereClause($filterCriteria, $query);
        $this->processPagination($filterCriteria, $query);
        $this->processGroupBy($filterCriteria, $query);
        $this->processOrderBy($filterCriteria, $query);
        return $params;
    }

    /**
     * Apply FilterCriteria conditions, whereClause, search and pagination to query.
     * Return a array of parameters needed to execute the query.
     *
     * @param FilterCriteria $filterCriteria
     * @param QueryBuilder   $query
     *
     * @return array
     */
    protected function processFilterConditionsForNext(FilterCriteria $filterCriteria, QueryBuilder $query) : array {
        $this->processSearch($filterCriteria, $query);
        $params = $this->processWhereClauseForNext($filterCriteria, $query);
        $this->processPagination($filterCriteria, $query);
        $this->processGroupBy($filterCriteria, $query);
        $this->processOrderBy($filterCriteria, $query);
        return $params;
    }
    
    public function getHttpRequest() {
        if ($this->httpRequest === null) {
            $this->httpRequest = Request::initFromGlobals();
        }

        return $this->httpRequest;
    }

    /**
     * Sets the FilterCriteria's search at the query where clause.
     *
     * @param FilterCriteria $filterCriteria The FilterCriteria object.
     * @param QueryBuilder $queryBuilder     The QueryBuilder object.
     */
    protected function processSearch(FilterCriteria $filterCriteria, QueryBuilder $query) {
        $search = $filterCriteria->getSearch();
        $columns = $filterCriteria->getSearchIn();
        $columns = (!empty($columns)) ? $columns : $this->dataSourceConfig->getColumns();
        $like = new \Zeedhi\Framework\DataSource\Operator\LikeI($this->dataSourceConfig);
        $this->getHttpRequest();
        
        $parameters = $this->httpRequest->getQueryParameters()->getAll();

        $searchInParams = isset($parameters['search_in_params']) ? json_decode($parameters['search_in_params'], true) : [];

        $searchJoinParams = isset($parameters['search_join']) ? json_decode(base64_decode($parameters['search_join']), true) : [];

        if (!is_array($searchJoinParams)) {
            $searchJoinParams = [$searchJoinParams];
        }

        $searchJoinColumnsKeys = array_keys($searchJoinParams);
        $remainingColumns = array_diff($columns, $searchJoinColumnsKeys);

        if (!empty($searchJoinParams)) {
            foreach ($searchJoinParams as $columnName => $columnValues) {
                if (!empty($columns) && in_array($columnName, $columns)) {
                    if (!empty($columnValues)) {
                        foreach ($columnValues as $innerKey => $value) {
                            $typeLike = "'%".$value."%'";
                            $query->orWhere($like->buildExpression($columnName, $typeLike));
                        }
                    } else if (empty($remainingColumns)) {
                        /*
                        Caso a coluna `$columnName`, nos parâmetros do searchJoin, não possua nenhum valor a
                        ser buscado, utiliza uma cláusula WHERE que sempre retornará FALSE, para a busca não
                        retornar nada (caso esta seja filtrada para considerar somente a coluna `$columnName`).
                        */
                        $query->orWhere('1 = 0');
                    }
                }
            }
        }

        foreach ($search as $value) {
            foreach ($remainingColumns as $column) {
                $typeLikeParam = isset($searchInParams[$column]) ? $searchInParams[$column] : 'A';
                $typeLike = $typeLikeParam === 'B' ? "'".$value."%'" : ($typeLikeParam === 'E' ? "'%".$value."'" : "'%".$value."%'");
                $query->orWhere($like->buildExpression($column, $typeLike));
            }
        }
    }

    /**
     * @param $params
     *
     * @return array
     */
    protected function inferTypes($params) {
        $types = array();
        foreach ($params as $name => $value) {
            $types[$name] = ParameterTypeInferer::inferType($value);
        }
        return $types;
    }

    /**
     * Return a DataSet with rows that match the given criteria.
     *
     * @param FilterCriteria $filterCriteria
     *
     * @return DataSet The result of the filter criteria.
     */
    public function findBy(FilterCriteria $filterCriteria) {
        $this->loadCurrentDataSource($filterCriteria);
        $rows = $this->retrieveRows($filterCriteria);
        $rows = $this->addIsNewColumn($rows);
        return new DataSet($filterCriteria->getDataSourceName(), $rows);
    }

    /**
     * Return a DataSet with rows and pagination that match the given criteria.
     *
     * @param FilterCriteria $filterCriteria
     *
     * @return DataSet The result of the filter criteria.
     */
    public function findByForNext(FilterCriteria $filterCriteria) : DataSet {
        $this->loadCurrentDataSource($filterCriteria);
        $rows = $this->retrieveRowsForNext($filterCriteria);
        $pagination = $this->getPagination($filterCriteria);
        return new DataSet($filterCriteria->getDataSourceName(), $rows, $pagination);
    }

    /**
     * Return the pagination array with total rows and the size
     * of actual dataset.
     *
     * @return array Pagination array
     */
    protected abstract function getPagination(FilterCriteria $filterCriteria) : array;

    /**
     * @param FilterCriteria $filterCriteria
     * @return array
     */
    protected abstract function retrieveRowsForNext(FilterCriteria $filterCriteria) : array;

    /**
     *  Checks if the request is to view an specific row.
     *
     * @param array $parameter
     *
     * @return bool
     */
    public function isViewRequest(array $parameters) : bool {
        $columns = $this->dataSourceConfig->getPrimaryKeyColumns();
        foreach($columns as $column) {
            if(!isset($parameters[$column])) return false;
        }
        return count($columns) > 0;
    }

    /**
     * Returns the data source configuration query parameters.
     *
     * @param FilterCriteria $filterCriteria The Filter criteria object.
     *
     * @return array
     */
    public function getQueryParametersFromJson(FilterCriteria $filterCriteria) : array {
        $this->loadCurrentDataSource($filterCriteria);
        return $this->dataSourceConfig->getParameters();
    }
}