<?php
namespace tests\Zeedhi\Framework\HTTP;

use UnexpectedValueException;

use Zeedhi\Framework\HTTP\Request;
use Zeedhi\Framework\HTTP\Parameter;

class RequestTest extends \PHPUnit\Framework\TestCase {

	public function testCreateRequest() {
		$request = self::create('http://pokemon:pokemon@example.net:9090/?foo=bar', 'GET');
		$this->assertInstanceOf('Zeedhi\Framework\HTTP\Request', $request, 'Its is expected an instance of Request');
		$this->assertEquals('/?foo=bar', $request->getRequestUri(), 'It is expected a request URI : /?foo=bar');
		$this->assertEquals(9090, $request->getPort());
		$this->assertEquals('example.net', $request->getHost());
		$this->assertFalse($request->isSecure());
		$this->differentUris();
	}

	protected function differentUris() {
		$request = self::create('https://[::1]/foo?bar=baz');
		$this->assertEquals('/foo?bar=baz', $request->getRequestUri(), 'It is expected a request URI : /foo?bar=baz');
		$this->assertEquals(443, $request->getPort());
		$this->assertEquals('[::1]', $request->getHttpHost());
		$this->assertTrue($request->isSecure());
	}

	public function testCreateFromGlobalsWithRawBody() {
		$_SERVER['REQUEST_METHOD'] = 'POST';
		$_SERVER['HTTP_USER_ID'] = 'bhlb9n2oq8lac3di';
		$_SERVER['CONTENT_TYPE'] = 'application/json';

		$request = Request::initFromGlobals();
		$this->assertInstanceOf('Zeedhi\Framework\HTTP\Request', $request, 'Its is expected an instance of Request');
		$this->assertEquals('POST', Request::getMethod(), 'It is expected a method POST');
		$this->assertEquals('', $request->getContent(), 'It is expected an content is empty');
		$this->assertEquals('bhlb9n2oq8lac3di', $request->getUserId(), 'It is expected userId equals "bhlb9n2oq8lac3di"');
		unset($_SERVER['REQUEST_METHOD'], $_SERVER['HTTP_USER_ID'], $_SERVER['CONTENT_TYPE']);
	}

	public function testCreateFromGlobals() {
		$_GET['foo1'] = 'bar1';
		$_POST['requestType'] = 'filter';
		$_COOKIE['foo3'] = 'bar3';
		$_FILES['foo4'] = array('bar4');
		$_SERVER['REQUEST_METHOD'] = 'POST';
		$_SERVER['HTTP_USER_ID'] = 'bhlb9n2oq8lac3di';

		$request = Request::initFromGlobals();
		$this->assertInstanceOf('Zeedhi\Framework\HTTP\Request', $request, 'Its is expected an instance of Request');
		$this->assertEquals('POST', Request::getMethod(), 'It is expected a method POST');
		$this->assertEquals('', $request->getContent(), 'It is expected an content is empty');
		$this->assertEquals('filter', $request->getRequestType(), 'It is expected a requestType: filter');
		$this->assertEquals('bhlb9n2oq8lac3di', $request->getUserId(), 'It is expected userId equals "bhlb9n2oq8lac3di"');
		unset($_GET['foo1'], $_POST['requestType'], $_COOKIE['foo3'], $_SERVER['REQUEST_METHOD'], $_SERVER['HTTP_USER_ID']);
	}

	public function testRequestWithRawBody() {
		$json = '{"jsonrpc":"2.0","method":"echo","id":7,"params":["Hello World"]}';
		$request = self::create('http://cdn.zeedhi.com/jsonrpc', 'POST', array(), array(), $json);
		$this->assertEquals($json, $request->getContent());
		$this->assertEquals(80, $request->getPort());
		$this->assertEquals('cdn.zeedhi.com', $request->getHttpHost());
		$this->assertFalse($request->isSecure());
	}

	public function testGetContentReturnsResource() {
		$req = new Request();
		$retVal = $req->getContent(true);
		$this->assertInternalType('resource', $retVal);
		$this->assertEquals("", fread($retVal, 1));
		$this->assertTrue(feof($retVal));
	}

	/**
	 * @dataProvider provideHostsToValidate
	 */
	public function testHostValidate($host, $isValid, $expectedHost = null, $expectedPort = null) {
		$request = self::create('/');
		$request->getHeaders()->set('host', $host);
		if ($isValid) {
			$this->assertSame($expectedHost ?: $host, $request->getHost());
			if ($expectedPort) {
				$this->assertSame($expectedPort, $request->getPort());
			}
		} else {
			$this->expectException(UnexpectedValueException::class);
			$this->expectExceptionMessage('Invalid Host');
			$request->getHost();
		}
	}

	public function testGetQueryParameters() {
		$parameters = array(
			'foo' => 'bar',
			'baz' => 'qux'
		);
		$request = new Request($parameters);

		$this->assertEquals(new Parameter($parameters), $request->getQueryParameters());
		$this->assertEquals(new Parameter(array()), $request->getRequestParameters());
	}

	public function testGetRequestParameters() {
		$parameters = array(
			'foo' => 'bar',
			'baz' => 'qux'
		);
		$request = new Request(array(), $parameters);

		$this->assertEquals(new Parameter(array()), $request->getQueryParameters());
		$this->assertEquals(new Parameter($parameters), $request->getRequestParameters());
	}

	public function provideHostsToValidate() {
		return array(
			array('.a', false),
			array('a..', false),
			array('a.', true),
			array("\xE9", false),
			array('[::1]', true),
			array('[::1]:80', true, '[::1]', 80),
			array(str_repeat('.', 101), false),
		);
	}

	/**
	 * Creates a Request based on a given URI and configuration.
	 *
	 * The information contained in the URI always take precedence
	 * over the other information (server and parameters).
	 *
	 * @param string     $uri        The URI
	 * @param string     $method     The HTTP method
	 * @param array      $parameters The query (GET) or request (POST) parameters
	 * @param array      $server     The server parameters ($_SERVER)
	 * @param mixed|null $content    The body of the raw request
	 *
	 *
	 * @return Request A Request instance
	 */
	public static function create($uri, $method = 'GET', $parameters = [], $server = [], $content = null) : Request {
		$server = array_replace(array(
			'SERVER_NAME' => '192.168.120.57',
			'SERVER_PORT' => 80,
			'HTTP_HOST' => '192.168.120.57',
			'HTTP_USER_AGENT' => 'Zeedhi/2.X',
			'HTTP_ACCEPT' => 'text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
			'HTTP_ACCEPT_LANGUAGE' => 'en-us,en;q=0.5',
			'HTTP_ACCEPT_CHARSET' => 'ISO-8859-1,utf-8;q=0.7,*;q=0.7',
			'REMOTE_ADDR' => '127.0.0.1',
			'SCRIPT_NAME' => '',
			'SCRIPT_FILENAME' => '',
			'SERVER_PROTOCOL' => 'HTTP/1.1',
			'REQUEST_TIME' => time(),
		), $server);

		$server['PATH_INFO'] = '';
		$server['REQUEST_METHOD'] = strtoupper($method);
		$_SERVER['REQUEST_METHOD'] = strtoupper($method);

		$components = parse_url($uri);
		if (isset($components['host'])) {
			$server['SERVER_NAME'] = $components['host'];
			$server['HTTP_HOST'] = $components['host'];
		}

		if (isset($components['scheme'])) {
			if ('https' === $components['scheme']) {
				$server['HTTPS'] = 'on';
				$server['SERVER_PORT'] = 443;
			} else {
				unset($server['HTTPS']);
				$server['SERVER_PORT'] = 80;
			}
		}

		if (isset($components['port'])) {
			$server['SERVER_PORT'] = $components['port'];
			$server['HTTP_HOST'] = $server['HTTP_HOST'] . ':' . $components['port'];
		}

		if (isset($components['user'])) {
			$server['PHP_AUTH_USER'] = $components['user'];
		}

		if (isset($components['pass'])) {
			$server['PHP_AUTH_PW'] = $components['pass'];
		}

		if (!isset($components['path'])) {
			$components['path'] = '/';
		}

        if (!isset($server['CONTENT_TYPE'])) {
            $server['CONTENT_TYPE'] = 'application/x-www-form-urlencoded';
        }

		if (strtoupper($method) === 'PATCH') {
            $request = $parameters;
            $query = [];
        } else {
            $request = [];
            $query = $parameters;
        }

		$queryString = '';
		if (isset($components['query'])) {
			parse_str(html_entity_decode($components['query']), $qs);
			if ($query) {
				$query = array_replace($qs, $query);
				$queryString = http_build_query($query, '', '&');
			} else {
				$query = $qs;
				$queryString = $components['query'];
			}
		} elseif ($query) {
			$queryString = http_build_query($query, '', '&');
		}

		$server['REQUEST_URI'] = $components['path'] . ($queryString ? '?' . $queryString : '');
		$server['QUERY_STRING'] = $queryString;
		$_SERVER['QUERY_STRING'] = $queryString;
		return new Request($query, $request, $server, $content);
	}

}
