<?php
namespace tests\Zeedhi\Framework\Controller;

use Zeedhi\Framework\Controller\CrudNext;
use Zeedhi\Framework\DataSource\Manager\IdProvider\ManagerImpl;
use Zeedhi\Framework\DataSource\Manager\IdProvider\Providers\SqlSequence;
use Zeedhi\Framework\DataSource\Manager\Doctrine;
use Doctrine\ORM\EntityManager;
use HumanRelation\Util\DataSource\NameProvider;
use Zeedhi\Framework\DataSource\ParameterBag;
use Zeedhi\Framework\Cache\Type\ArrayImpl;
use Zeedhi\Framework\DTO;
use Zeedhi\Framework\DataSource\DataSet;
use Zeedhi\Framework\DataSource\FilterCriteria;
use Zeedhi\Framework\Routing\Router;

class CrudNextTest extends \PHPUnit\Framework\TestCase {

    const USER_ID = 'USR_ORG_20';
    /** @var ManagerImpl */
    protected $dataSourceManager;
    /** @var EntityManager */
    protected $entityManager;
    /** @var CrudNext */
    protected $crudNext;
    /** @var ParameterBag */
    protected $parameterBag;

    public function setUp() {
        /** @var EntityManager $entityManager */
        $entityManager = \HumanRelation\Util\EntityManagerFactory::createWithOracleConnection();
        $this->entityManager = $entityManager;
        $this->parameterBag = new ParameterBag(new ArrayImpl());
        $nameProvider = new NameProvider();
        $doctrineManager = new Doctrine\ManagerImpl($entityManager, $nameProvider, $this->parameterBag);
        $idProvider = new SqlSequence($this->entityManager->getConnection());
        $this->dataSourceManager = new ManagerImpl($nameProvider, $doctrineManager, $idProvider);
        $this->createSavePoint();
        $this->crudNext = new CrudNext($this->dataSourceManager, $nameProvider, $this->parameterBag);
    }

    protected function tearDown() {
        $this->backToSavePoint();
        parent::tearDown();
    }

    protected function createSavePoint() {
        $this->entityManager->beginTransaction();
    }

    protected function backToSavePoint() {
        $this->entityManager->rollback();
    }

    public function testSave() {
        $row = [
            "REGION_NAME" => "Antarctica"
        ];
        $dataSet = new DataSet("regions", [ $row ]);
        $request = new DTO\Request\DataSet($dataSet, Router::METHOD_POST, "/regions", self::USER_ID);
        $response = new DTO\Response();
        $this->crudNext->persist($request, $response);
        /** @var DTO\Next\Response $nextResponse */
        $nextResponse = $response->getZeedhiNext();
        $this->assertInstanceOf("Zeedhi\\Framework\\DTO\\Next\\Response", $nextResponse);
        $this->assertIsNumeric($nextResponse->getRow()['REGION_ID']);
    }

    public function testUpdate() {
        $row = [
            "REGION_ID"   => 1,
            "REGION_NAME" => "Antartica"
        ];
        $dataSet = new DataSet("regions", [ $row ]);
        $request = new DTO\Request\DataSet($dataSet, Router::METHOD_PUT, "/regions/1", self::USER_ID);
        $response = new DTO\Response();
        $this->crudNext->persist($request, $response);
        /** @var DTO\Next\Response $nextResponse */
        $nextResponse = $response->getZeedhiNext();
        $this->assertInstanceOf("Zeedhi\\Framework\\DTO\\Next\\Response", $nextResponse);
        $this->assertEquals(["REGION_ID" => 1], $nextResponse->getRow());
        /** @var Regions $region */
        $region = $this->entityManager->find("\\HumanRelation\\Entities\\Regions", 1);
        $this->assertInstanceOf("\\HumanRelation\\Entities\\Regions", $region, "A region with id 1 must exist.");
        $this->assertEquals("Antartica", $region->getRegionName(), "Name of region with id 6 must be 'Antartica'.");
    }

    public function testFind() {
        $filterCriteria = new FilterCriteria("regions");
        $filterCriteria->addCondition("REGION_NAME", "Americas");
        $request = new DTO\Request\Filter($filterCriteria, Router::METHOD_GET, "/regions?REGION_NAME=Americas", self::USER_ID);
        $response = new DTO\Response();
        $this->crudNext->find($request, $response);
        /** @var DTO\Next\Response $nextResponse */
        $nextResponse = $response->getZeedhiNext();
        $this->assertInstanceOf("Zeedhi\\Framework\\DTO\\Next\\Response", $nextResponse);
        /** @var DataSet[] $dataSet */
        $dataSet = $nextResponse->getDataSet();
        $this->assertInstanceOf("\\Zeedhi\\Framework\\DataSource\\DataSet", $dataSet, "Return must be a instance of DataSet.");
        $this->assertCount(1, $dataSet->getRows(), "Must return 1 rows.");
        foreach($dataSet->getRows() as $row) {
            $this->assertEquals("Americas", $row["REGION_NAME"], "All retrieved regions should name 'Americas'.");
        }
    }

    public function testFindUsingDataSourceConfigurationQuery() {
        $filterCriteria = new FilterCriteria("employees_with_query");
        $filterCriteria->addCondition("FIRST_NAME", FilterCriteria::EQ, '%lan%');
        $filterCriteria->addCondition("SALARY", FilterCriteria::EQ, 3100);
        $request = new DTO\Request\Filter($filterCriteria, Router::METHOD_GET, "/employees_with_query?SALARY=3100&search=lan&search_in=FIRST_NAME", self::USER_ID);
        $response = new DTO\Response();
        $this->crudNext->find($request, $response);
        /** @var DTO\Next\Response $nextResponse */
        $nextResponse = $response->getZeedhiNext();
        /** @var DataSet[] $dataSet */
        $dataSet = $nextResponse->getDataSet();
        $this->assertCount(1, $dataSet->getRows());
    }

    public function testView() {
        $filterCriteria = new FilterCriteria("regions");
        $request = new DTO\Request\Filter($filterCriteria, Router::METHOD_GET, "/regions/1", self::USER_ID);
        $request->setParameter('REGION_ID', 1);
        $response = new DTO\Response();
        $this->crudNext->find($request, $response);
        /** @var DTO\Next\Response $nextResponse */
        $nextResponse = $response->getZeedhiNext();
        $this->assertInstanceOf("Zeedhi\\Framework\\DTO\\Next\\Response", $nextResponse);
        /** @var DataSet[] $dataSet */
        $dataSet = $nextResponse->getDataSet();
        $row = $nextResponse->getRow();
        $expectedRow = ['REGION_NAME' => 'Europe', 'REGION_ID' => 1];
        $this->assertEquals($expectedRow, $row, "Must return 1 row.");
        $this->assertEquals("Europe", $row["REGION_NAME"], "All retrieved regions should name 'Europe'.");
    }

    public function testDelete() {
        $filterCriteria = new FilterCriteria("countries");
        $filterCriteria->addCondition("COUNTRY_ID", "AR");
        $request = new DTO\Request\Filter($filterCriteria, Router::METHOD_DELETE, "/countries/AR", self::USER_ID);
        $response = new DTO\Response();
        $this->crudNext->delete($request, $response);
        /** @var DTO\Next\Response $nextResponse */
        $nextResponse = $response->getZeedhiNext();
        $this->assertInstanceOf("Zeedhi\\Framework\\DTO\\Next\\Response", $nextResponse);
        /** @var DTO\Row $returnedDataSet */
        $returnedRow = $nextResponse->getRow();
        $this->assertEquals(["COUNTRY_ID" => "AR"], $returnedRow);
        /** @var Countries $country */
        $country = $this->entityManager->find("\\HumanRelation\\Entities\\Countries", "AR");
        $this->assertNull($country, "Country 'Argentina' must not exists anymore.");
    }

    public function testSaveWithParameters() {
        $row = [ "REGION_NAME" => "Antarctica" ];
        $dataSet = new DataSet("regions", [ $row ]);
        $request = new DTO\Request\DataSet($dataSet, Router::METHOD_POST, "/regions/6", self::USER_ID);
        $request->setParameter("REGION_ID", 6);
        $response = new DTO\Response();
        $this->crudNext->persist($request, $response);
        /** @var DTO\Next\Response $nextResponse */
        $nextResponse = $response->getZeedhiNext();
        $this->assertInstanceOf("Zeedhi\\Framework\\DTO\\Next\\Response", $nextResponse);
        $this->assertEquals(["REGION_ID" => 6], $nextResponse->getRow());
    }

    public function testSaveError() {
        $row = [
            "REGION_NAME" => "Antarctica Antarctica Antarctica Antarctica Antarctica Antarctica Antarctica Antarctica"
        ];
        $dataSet = new DataSet("regions", [ $row ]);
        $request = new DTO\Request\DataSet($dataSet, Router::METHOD_POST, "/regions?REGION_ID=6&REGION_NAME=Antarctica%20Antarctica%20Antarctica%20Antarctica%20Antarctica%20Antarctica%20Antarctica%20Antarctica", self::USER_ID);
        $response = new DTO\Response();
        $this->crudNext->persist($request, $response);
        /** @var DTO\Next\Response $nextResponse */
        $this->assertNull($response->getZeedhiNext(), "No DTO\Next\Response sets must be returned");
        $error = $response->getError();
        $this->assertInstanceOf('\Zeedhi\Framework\DTO\Response\Error', $error);
        $this->assertNotEmpty($error->getMessage(), 'Error message should not be empty!');
        $this->assertNotEmpty($error->getStackTrace(), 'Stack trace should not be empty!');
    }
}