<?php
namespace Zeedhi\Framework\Controller;

use Zeedhi\Framework\DTO;
use Zeedhi\Framework\Controller\Simple;
use Zeedhi\Framework\DataSource\Manager;
use Zeedhi\Framework\DataSource\FilterCriteria;
use Zeedhi\Framework\DataSource\DataSet;
use Zeedhi\Framework\DataSource\ParameterBag;
use Zeedhi\Framework\DataSource\Manager\Doctrine\NameProvider;

class CrudNext extends Simple {

    /** @var Manager The Entity Manager used to interact with Database. */
    private $dataSourceManager;
    /** @var NameProvider */
    private $nameProvider;
    /** @var ParameterBag */
    private $parameterBag;

    /**
     * Constructor...
     *
     * @param Manager $dataSourceManager The DataSourceManager implementation used by your product.
     * @param NameProvider $nameProvider The NameProvider used to discover ClassNames.
     * @param ParameterBag $parameterBag The ParameterBag used to keep query parameters.
     */
    public function __construct(Manager $dataSourceManager, NameProvider $nameProvider, ParameterBag $parameterBag) {
        $this->dataSourceManager = $dataSourceManager;
        $this->nameProvider = $nameProvider;
        $this->parameterBag = $parameterBag;
    }

    /**
     * Action used for insert new rows and update already existent.
     *
     * @param DTO\Request\DataSet $request  A DTO\Request\DataSet with all rows to be persisted..
     * @param DTO\Response   $response A DTO\Response, mostly of cases will only add messages.
     */
    public function persist(DTO\Request\DataSet $request, DTO\Response $response) {
        try {
            $dataSet = $request->getDataSet();
            $parameters = $request->getParameters();
            $dataSet = $this->addPropertiesOnDataSet($dataSet, $parameters);
            $dataSet = $this->dataSourceManager->populateDataSet($dataSet);
            $isNew = $this->getIsNew($request->getMethod());
            $persistedRow = $this->dataSourceManager->persistForNext($dataSet, $isNew);
            $dataSourceName = $dataSet->getDataSourceName();
            $next = new DTO\Next\Response(
                $dataSourceName,
                $this->getPrimaryKeyColumns($dataSourceName)
            );
            $isDataSet = count($dataSet->getRows()) > 1;
            $this->setPersistedDataOnResponse($next, $dataSourceName, $persistedRow, $isDataSet);
            $response->setZeedhiNext($next);
        } catch(\Exception $e) {
            $response->setError(new DTO\Response\Error($e->getMessage(), $e->getCode(), $e->getTraceAsString()));
        }
    }

    /**
     * Action used for find rows that match with given FilterCriteria
     *
     * @param DTO\Request\Filter $request  A DTO\Request with FilterCriteria to match the rows in DataSourceManager.
     * @param DTO\Response  $response A DTO\Response to be added with DataSets of matched rows.
     */
    public function find(DTO\Request\Filter $request, DTO\Response $response) {
        try {
            $filterCriteria = $request->getFilterCriteria();
            $parameters = $request->getParameters();
            $this->addConditionsOnFilter($filterCriteria, $parameters);
            $dataSet = $this->dataSourceManager->findByForNext($filterCriteria);
            $dataSourceName = $dataSet->getDataSourceName();
            $next = new DTO\Next\Response(
                $dataSourceName,
                $this->getPrimaryKeyColumns($dataSourceName)
            );

            if($this->dataSourceManager->isViewRequest($parameters) && ($rows = $dataSet->getRows()))
                $next->setRow(reset($rows));
            else
                $next->setDataSet($dataSet);

            $response->setZeedhiNext($next);
        } catch(\Exception $e) {
            $response->setError(new DTO\Response\Error($e->getMessage(), $e->getCode(), $e->getTraceAsString()));
        }
    }

    /**
     * Action used for delete some rows.
     *
     * @param DTO\Request\Filter $request  A DTO\Request with DataSet of all rows to be deleted..
     * @param DTO\Response  $response A DTO\Response, mostly of cases will only add messages.
     */
    public function delete(DTO\Request\Filter $request, DTO\Response $response) {
        try {
            $filterCriteria = $request->getFilterCriteria();
            $parameters = $request->getParameters();
            $this->addConditionsOnFilter($filterCriteria, $parameters);
            $row = $this->buildRowWithConditions($filterCriteria->getConditions());
            $dataSourceName = $filterCriteria->getDataSourceName();
            $deletedRow = $this->dataSourceManager->deleteForNext(new DataSet($dataSourceName, [ $row ]));
            $next = new DTO\Next\Response(
                $dataSourceName,
                $this->getPrimaryKeyColumns($dataSourceName)
            );
            $next->setRow($deletedRow);
            $response->setZeedhiNext($next);
        } catch(\Exception $e) {
            $response->setError(new DTO\Response\Error($e->getMessage(), $e->getCode(), $e->getTraceAsString()));
        }
    }

    /**
     * Add new properties on DataSet.
     *
     * @param DataSet $dataSet
     * @param array $properties
     *
     * @return DataSet
     */
    private function addPropertiesOnDataSet(DataSet $dataSet, array $properties) {
        $dataSourceName = $dataSet->getDataSourceName();
        $rows = $dataSet->getRows();
        foreach ($properties as $key => $value) {
            $rows[0][$key] = $value;
        }
        return new DataSet($dataSourceName, $rows);
    }

    /**
     * Add conditions on Filter Criteria.
     *
     * @param FilterCriteria $filterCriteria
     * @param array $properties
     */
    private function addConditionsOnFilter(FilterCriteria $filterCriteria, array $properties) {
        foreach($properties as $key=>$property) {
            $filterCriteria->addCondition($key, (string)$property);
        }
    }

    /**
     * Set the result of save on response.
     *
     * @param DTO\Response $response
     * @param string $dataSourceName
     * @param array $persistedData
     */
    private function setPersistedDataOnResponse(DTO\Next\Response $response, string $dataSourceName, array $persistedData, bool $isDataSet) {
        if($isDataSet) {
            $response->setDataSet(new DataSet($dataSourceName, $persistedData));
        } else {
            $response->setRow($persistedData);
        }
    }

    /**
     * Get primary keys from a Data Source
     *
     * @param string $dataSourceName
     *
     * @return array
     */
    private function getPrimaryKeyColumns(string $dataSourceName) : array {
        return $this->nameProvider->getDataSourceByName($dataSourceName)->getPrimaryKeyColumns();
    }

    /**
     * Build a new row with condition from filter criteria
     *
     * @param array $conditions Conditions from FilterCriteria
     *
     * @return array Row
     */
    private function buildRowWithConditions(array $conditions) : array {
        foreach($conditions as $condition) {
            $row[$condition['columnName']] = $condition['value'];
        }
        return $row;
    }

    /**
     * Returns true if the request is to save or false if to update.
     *
     * @param string $method The requested method.
     *
     * @return bool Is new
     */
    private function getIsNew(string $method) : bool {
        return $method == 'POST';
    }
}