<?php
namespace Zeedhi\Framework\HTTP\Next;

use Zeedhi\Framework\HTTP\Request;
use Zeedhi\Framework\Service\Next;
use Zeedhi\Framework\DTO;
use Zeedhi\Framework\DataSource\FilterCriteria;
use Zeedhi\Framework\DataSource\DataSet;
use Zeedhi\Framework\Routing\Next\RouteFactory;
use Zeedhi\Framework\DependencyInjection\InstanceManager;

class RequestFactory {

    /** @var Request */
    private $request;
    /** @var string */
    private $baseUri;
    /** @var Next */
    private $next;
    /** @var RouteFactory */
    private $routeFactory;

    /**
     * Constructor
     *
     * @param Request $request The Request object from HTTP globals.
     */
    public function __construct(Request $request, string $baseUri) {
        $nameProvider = InstanceManager::getInstance()->getService('nameProvider');
        $this->request = $request;
        $this->baseUri = $baseUri;
        $this->next = new Next($nameProvider);
        $this->routeFactory = new RouteFactory($nameProvider);
    }

    /**
     * Builds the Request.
     *
     * @return DTO\Request\Filter|DTO\Request\DataSet
     *
     * @throws \Exception
     */
    public function buildRequest() {
        $method = Request::getMethod();
        $cleanUri = $this->removeBaseUri($this->request->getRequestUri());
        $cleanUri = $this->removeQueryFromUri($cleanUri);
        $arrayUri = Next::buildArrayUri($cleanUri);
        $dataSourceName = $this->next->getDataSourceName($arrayUri);
        $parameters = array_merge(
            $this->request->getQueryParameters()->getAll(),
            $this->request->getRequestParameters()->getAll()
        );
        $userId = $this->request->getUserId();
        if($method == 'GET' || $method == 'DELETE') {
            $filterCriteria = $this->buildFilterCriteria($parameters, Request::getQueryString(), $dataSourceName);
            $dtoRequest = new DTO\Request\Filter($filterCriteria, $method, $cleanUri, $userId);
        } elseif($parameters) {
            $dataSet = $this->buildDataSet($parameters, $dataSourceName);
            $dtoRequest = new DTO\Request\DataSet($dataSet, $method, $cleanUri, $userId);
        } else {
            throw new \Exception("No data found to be set or update!");
        }
        $this->routeFactory->buildRoute($cleanUri, $method);
        return $dtoRequest;
    }

    /**
     * Builds the FilterCriteria.
     *
     * @param array  $parameters     The request parameters.
     * @param array  $arrayUri       The array with the elements of uri (without query).
     * @param array  $queryString    The request query string
     * @param string $dataSourceName The data source name
     *
     * @return FilterCriteria
     */
    private function buildFilterCriteria(array $parameters, string $queryString, string $dataSourceName) : FilterCriteria {
        $filterCriteria = new FilterCriteria($dataSourceName);
        $this->buildOrderBy($queryString, $parameters, $filterCriteria);
        $this->buildSearch($parameters, $filterCriteria);
        $this->buildSearchIn($queryString, $parameters, $filterCriteria);
        $this->buildPagination($parameters, $filterCriteria);
        $this->buildConditions($queryString, $parameters, $filterCriteria);
        return $filterCriteria;
    }

    /**
     * Builds the DataSet.
     *
     * @param array  $parameters     The request parameters.
     * @param string $dataSourceName The data source name for dataset.
     *
     * @return DataSet
     */
    private function buildDataSet(array $parameters, string $dataSourceName) : DataSet {
        return new DataSet($dataSourceName, [$parameters]);
    }

    /**
     * Builds the orderBy for the FilterCriteria object.
     *
     * @param string $queryString           The request query string.
     * @param array  $parameters            The request parameters.
     * @param FilterCriteria $filterCriteria The FilterCriteria object.
     */
    private function buildOrderBy(string $queryString, array &$parameters, FilterCriteria $filterCriteria) {
        if(preg_match_all("/(?:(?<=\border=))\w+\.\w+/", $queryString, $orderBy)) {
            foreach(reset($orderBy) as $order) {
                $orderArray = explode('.', $order);
                $filterCriteria->addOrderBy($orderArray[0], $orderArray[1]);
                unset($parameters["order"]);
            }
        }
    }

    /**
     * Builds the search for the FilterCriteria object.
     *
     * @param array $parameteters            The request parameters.
     * @param FilterCriteria $filterCriteria The FilterCriteria object.
     */
    private function buildSearch(array &$parameters, FilterCriteria $filterCriteria) {
        if(isset($parameters["search"])){
            $values = explode(" ", $parameters["search"]);
            $value = '%' . implode('%', $values) . '%';
            $filterCriteria->addSearch($value);
            unset($parameters['search']);
        }
    }

    /**
     * Builds the searchIn for the FilterCriteria object.
     *
     * @param string $queryString             The request query string.
     * @param array  $parameteters            The request parameters.
     * @param FilterCriteria $filterCriteria The FilterCriteria object.
     */
    private function buildSearchIn(string $queryString, array &$parameters, FilterCriteria $filterCriteria) {
        if(preg_match_all("/(?:(?<=\bsearch_in=))\w+/", $queryString, $searchIn)) {
            $filterCriteria->setSearchIn(reset($searchIn));
            unset($parameters['search_in']);
        }
    }

    /**
     * Builds the pagination for the FilterCriteria object.
     *
     * @param array $parameteters            The request parameters.
     * @param FilterCriteria $filterCriteria The FilterCriteria object.
     */
    private function buildPagination(array &$parameters, FilterCriteria $filterCriteria) {
        if(isset($parameters["page"])) {
            $filterCriteria->setPage(intval($parameters["page"]));
            unset($parameters["page"]);
            if(isset($parameters["limit"])) {
                $filterCriteria->setPageSize(intval($parameters["limit"]));
                unset($parameters["limit"]);
            }
        }
    }

    /**
     * Builds the conditions for the FilterCriteria object.
     *
     * @param string $queryString             The request query string.
     * @param array  $parameteters            The request parameters.
     * @param FilterCriteria $filterCriteria The FilterCriteria object.
     */
    private function buildConditions(string $queryString, array &$parameters, FilterCriteria $filterCriteria) {
        if (preg_match_all("/(?:(?<=\bin=))\w+/", $queryString, $in)) {
            foreach(reset($in) as $column) {
                $filterCriteria->addCondition($column, $parameters[$column]);
            }
        }
    }

    /**
     * Removes base URI from received URI.
     *
     * @param string $uri Requested URI.
     *
     * @return string URI without base URi.
     */
    protected function removeBaseUri(string $uri) : string {
        $this->baseUri = rtrim($this->baseUri, ' /');
        return preg_replace('#(.*)(' . $this->baseUri . ')#', '', $uri);
    }

    /**
     * Removes query string from uri.
     *
     * @param string $uri
     *
     * @return string
     */
    private function removeQueryFromUri(string $uri) : string {
        $uri = explode('?', $uri);
        return $uri[0];
    }
}
