<?php
namespace Zeedhi\Framework\HTTP;
use Zeedhi\Framework\Routing\Router;

/**
 * Class Request
 *
 * Request represents an HTTP request.
 *
 * @package Zeedhi\Framework\HTTP
 */
class Request {
	/**
	 * Query string parameters ($_GET)
	 *
	 * @var Parameter
	 */
	protected $query;
	/**
	 * Request body parameters ($_POST)
	 *
	 * @var Parameter
	 */
	protected $request;
	/**
	 * Server and execution environment parameters ($_SERVER)
	 *
	 * @var Parameter
	 */
	protected $server;

	/**
	 * Headers (taken from the $_SERVER).
	 *
	 * @var Header
	 */
	protected $headers;

	/**
	 * @var string
	 */
	protected $content;

	public function __construct(array $query = array(), array $request = array(), array $server = array(), $content = null) {
		$this->query = new Parameter($query);
		$this->request = new Parameter($request);
		$this->server = new Server($server);
		$this->headers = new Header($this->server->getHeaders());
		$this->content = $content;
	}


	/**
	 * Gets a "parameter" value.
	 *
	 * @param string $key     the key
	 * @param mixed  $default the default value
	 * @param bool   $deep    is parameter deep in multidimensional array
	 *
	 * @return mixed
	 */
	public function get($key, $default = null, $deep = false) {
		return $this->query->get($key, $this->request->get($key, $default, $deep), $deep);
	}

	/**
	 * Gets the request "intended" method.
	 *
	 * @return string
	 */
	public static function getMethod() : string {
		return strtoupper(isset($_SERVER['REQUEST_METHOD']) ? $_SERVER['REQUEST_METHOD'] : '');
	}

	/**
	 * Returns the requested URI.
	 *
	 * @return string The raw URI (i.e. not urldecoded)
	 */
	public function getRequestUri() : string {
		return $this->server->get('REQUEST_URI');
	}

	/**
	 * Returns the request Type.
	 *
	 * @return string|null
	 */
	public function getRequestType() : ?string {
		return $this->request->get('requestType');
	}

	/**
	 * Retrieves the request type by method.
	 *
	 * @param string $method The request method.
	 * @return string
	 */
	protected function getRequestTypeByMethod(string $method) : string {
        return $method === 'GET' ? 'FilterData' : 'DataSet';
	}

	/**
	 * Returns the User-Id of the request header.
	 *
	 * @return string|null
	 */
	public function getUserId() {
		return $this->headers->get('User-Id');
	}

	/**
	 * Creates a new request with values from PHP's super globals.
	 *
	 * @return Request A new request
	 */
	public static function initFromGlobals() : Request {
		$params = [];
		$method = self::getMethod();

		if($method === Router::METHOD_PUT) {
			parse_str(file_get_contents("php://input"), $params);
		} if($method === Router::METHOD_GET || $method === Router::METHOD_DELETE) {
			if(array_key_exists('requestType', $_GET))
				$params['requestType'] = $_GET['requestType'];
		} else {
			$params = $_POST;
		}

		$request = new static($_GET, $params, $_SERVER);

		if (strpos($request->getContentType(), 'application/json') !== false) {
			$data = json_decode($request->getContent(), true);
			$request->request = new Parameter((array)$data);
		}

		return $request;
	}

	/**
	 * Returns the request body content.
	 *
	 * @param bool $asResource If true, a resource will be returned
	 *
	 * @return string|resource The request body content or a resource to read the body stream.
	 */
	public function getContent($asResource = false) {
		if (!$this->content) {
			if ($asResource) {
				$this->content = fopen('php://input', 'rb');
			} else {
				$this->content = file_get_contents('php://input');
			}
		}
		return $this->content;
	}

	/**
	 * Gets the format associated with the request.
	 *
	 * @return string|null The format (null if no content type is present)
	 */
	public function getContentType() {
		return $this->headers->get('Content-Type');
	}

	/**
	 * Returns the headers taken from $_SERVER
	 *
	 * @return Header
	 */
	public function getHeaders() {
		return $this->headers;
	}

	/**
	 * Checks whether the request is secure or not.
	 *
	 * @return bool
	 */
	public function isSecure() {
		$https = $this->server->get('HTTPS');
		return !empty($https) && 'off' !== strtolower($https);
	}

	/**
	 * Gets the request's scheme.
	 *
	 * @return string
	 */
	public function getScheme() {
		return $this->isSecure() ? 'https' : 'http';
	}


	/**
	 * Returns the port on which the request is made.
	 *
	 * @return string
	 */
	public function getPort() {
		if ($host = $this->headers->get('HOST')) {
			if ($host[0] === '[') {
				$pos = strpos($host, ':', strrpos($host, ']'));
			} else {
				$pos = strrpos($host, ':');
			}
			if (false !== $pos) {
				return intval(substr($host, $pos + 1));
			}
			return 'https' === $this->getScheme() ? 443 : 80;
		}
		return $this->server->get('SERVER_PORT');
	}

	/**
	 * Returns the host name.
	 *
	 * @return string
	 *
	 * @throws \UnexpectedValueException when the host name is invalid
	 */
	public function getHost() {
		$host = $this->headers->get('HOST');
		if (!$host) {
			$host = $this->server->get('SERVER_NAME');
			if (!$host) {
				$host = $this->server->get('SERVER_ADDR', '');
			}
		}
		$host = strtolower(preg_replace('/:\d+$/', '', trim($host)));
		if ($host && '' !== preg_replace('/(?:^\[)?[a-zA-Z0-9-:\]_]+\.?/', '', $host)) {
			throw new \UnexpectedValueException(sprintf('Invalid Host "%s"', $host));
		}
		return $host;
	}

	/**
	 * Returns the HTTP host being requested.
	 *
	 * The port name will be appended to the host if it's non-standard.
	 *
	 * @return string
	 */
	public function getHttpHost() {
		$scheme = $this->getScheme();
		$port = $this->getPort();
		if (('http' == $scheme && $port == 80) || ('https' == $scheme && $port == 443)) {
			return $this->getHost();
		}
		return $this->getHost() . ':' . $port;
	}

	/**
	 * Gets the scheme and HTTP host.
	 *
	 * If the URL was called with basic authentication, the user
	 * and the password are not added to the generated string.
	 *
	 * @return string The scheme and HTTP host
	 */
	public function getSchemeAndHttpHost() {
		return $this->getScheme() . '://' . $this->getHttpHost();
	}

	/**
	 * Gets the query parameters
	 *
	 * @return Parameter
	 */
	public function getQueryParameters() {
		return $this->query;
	}

	/**
	 * Gets the request parameters
	 *
	 * @return Parameter
	 */
	public function getRequestParameters() {
		return $this->request;
	}

	/**
	 * Gets the request query string.
	 *
	 * @return string Query string.
	 */
	public static function getQueryString() : string {
		return $_SERVER['QUERY_STRING'];
	}

}