<?php
namespace Zeedhi\Framework\Routing;
use Zeedhi\Framework\DTO\Request;

/**
 * Class Router
 *
 * Responsible for find appropriated controller for determined route.
 *
 * @package Zeedhi\Framework\Routing
 */
class Router {

    /** @const string */
    public const METHOD_DELETE = 'DELETE';
    /** @const string */
    public const METHOD_POST = 'POST';
    /** @const string */
    public const METHOD_GET = 'GET';
    /** @const string */
    public const METHOD_PUT = 'PUT';

    /** @var Route[] $post */
    protected static $post = [];
    /** @var Route[] $get */
    protected static $get = [];
    /** @var Route[] $put */
    protected static $put = [];
    /** @var Route[] $delete */
    protected static $delete = [];
    /** @var Parser */
    protected $parser;
    /** @var boolean */
    protected $routesRead = false;

    /**
     * @param \Zeedhi\Framework\Routing\Parser $parser
     */
    public function setParser(Parser $parser) {
        $this->parser = $parser;
    }

    public function readRoutes() {
        if($this->routesRead) return;
        $this->parser->parseFile($this);
        $this->routesRead = true;
    }

    /**
     * Method that resolves the received URI.
     * Returns an array with the controller name and the method name
     *
     * @param Request $request Contains the method and url called.
     *
     * @throws Exception Route does not exist.
     *
     * @return array Where first position determine controller name, in DI Container, and second position his method.
     */
    public function resolveRoute(Request $request) : array {
        $routes = self::getRoutesForRequestMethod($request->getMethod());
        foreach ($routes as $route) {
            if ($route->matchRequest($request)) {
                return [$route->getController(), $route->getControllerMethod()];
            }
        }

        throw Exception::routeDoesNotExist($request->getRoutePath());
    }

    /**
     * Function that add a new POST route
     *
     * @param Route $route
     */
    public static function post(Route $route) {
        self::$post[] = $route;
    }

    /**
     * Function that add a new GET route
     *
     * @param Route $route
     */
    public static function get(Route $route) {
        self::$get[] = $route;
    }

    /**
     * Function that add a new PUT route
     *
     * @param Route $route
     */
    public static function put(Route $route) {
        self::$put[] = $route;
    }

    /**
     * Function that add a new DELETE route
     *
     * @param Route $route
     */
    public static function delete(Route $route) {
        self::$delete[] = $route;
    }

    /**
     * Function that add a new route with ANY HTTP method
     *
     * @param Route $route
     */
    public function any(Route $route) {
        self::post($route);
        self::put($route);
        self::delete($route);
        self::get($route);
    }

    /**
     * @param Request $request
     * @return Route[]
     * @throws Exception
     */
    protected static function getRoutesForRequestMethod(string $method) : array {
        /** @var Route[] $routes */
        switch ($method) {
            case self::METHOD_POST:
                $routes = self::$post;
                break;
            case self::METHOD_GET:
                $routes = self::$get;
                break;
            case self::METHOD_PUT:
                $routes = self::$put;
                break;
            case self::METHOD_DELETE:
                $routes = self::$delete;
                break;
            default:
                throw Exception::invalidMethod($method);
        }
        return $routes;
    }

    /**
     * Function that add a route.
     *
     * @param Route $route
     * @throws Exception Throws invalid method exception.
     */
    public static function addRoute(Route $route) {
        foreach($route->getMethods() as $method) {
            switch ($method) {
                case self::METHOD_POST:
                    self::post($route);
                    break;
                case self::METHOD_GET:
                    self::get($route);
                    break;
                case self::METHOD_PUT:
                    self::put($route);
                    break;
                case self::METHOD_DELETE:
                    self::delete($route);
                    break;
                default:
                    throw Exception::invalidMethod($method);
            }
        }
    }

    /**
     * Returns a Route if these exist and NULL if not.
     *
     * @param string $routePath Requested route path.
     * @param string $method    Requested method.
     *
     * @return Route|null
     */
    public static function getRouteByPath(string $routePath, string $method) {
        $routes = self::getRoutesForRequestMethod($method);
        foreach($routes as $route) {
            if($route->matchRoutePath($routePath))
                return $route;
        }
    }
}