<?php
namespace Zeedhi\Framework\DataSource;

use Zeedhi\Framework\Util\QuerySelector;
/**
 * Class Configuration
 *
 * @package Zeedhi\Framework\DataSource
 */
class Configuration {

    /** @var string */
    protected $name;
    /** @var string */
    protected $tableName;
    /** @var array */
    protected $columns;
    /** @var array */
    protected $primaryKeyColumns = [];
    /** @var string */
    protected $sequentialColumn;
    /** @var array */
    protected $relations = [];
    /** @var string */
    protected $query = '';
    /** @var string */
    protected $dataColumnsByColumn;
    /** @var array */
    protected $orderBy = [];
    /** @var array */
    protected $groupBy = [];
    /** @var int Zero (0) value will be treated as unlimited. */
    protected $resultSetLimit = 0;
    /** @var array */
    protected $conditions = [];
    /** @var array $parameters Parameters that are present in the query. */
    protected $parameters = [];
    /** @var array $defaults Default values for parameters when they have not been set. */
    protected $defaults = [];
    
    /** @var array */
    protected $columnType = [];
    /** @var array $metaData Json Metadata. */
    protected $metaData = [];
    /** @var array $driverName */
    protected $driverName = '';

    /**
     * Constructor.
     * @param string $name      The data source name.
     * @param array  $columns   The data source columns.
     */
    public function __construct(string $name, array $columns) {
        $this->name = $name;
        $this->setColumns($columns);
    }

    /**
     * Sets passed primary key columns that are contained by the column list.
     *
     * @param array $primaryKeyColumns The primary key columns list.
     *
     * @throws Exception If the column is not found os columns list.
     */
    public function setPrimaryKeyColumns(array $primaryKeyColumns) {
        foreach ($primaryKeyColumns as $pkColumn) {
            if (!in_array($pkColumn, $this->columns)) {
                throw Exception::pkColumnNotFoundInColumnList($pkColumn, $this->name);
            }
        }
        $this->primaryKeyColumns = $primaryKeyColumns;
    }

    /**
     * Sets the sequential column passed if it's referenced as primary key.
     *
     * @param string $sequentialColumn The sequential column name.
     *
     * @throws Exception If the columns is not referenced by a primary key.
     */
    protected function setSequentialColumn(string $sequentialColumn) {
        if (in_array($sequentialColumn, $this->primaryKeyColumns)) {
            $this->sequentialColumn = $sequentialColumn;
        } else {
            throw Exception::sequentialColumnMustBeAPkColumn($this->name, $sequentialColumn);
        }
    }

    /**
     * Retrieves the data source configuration file as array.
     *
     * @param string $dirLocation       The file dir location.
     * @param string $dataSourceName    The data source file name.
     *
     * @return array
     *
     * @throws Exception With invalid file name.
     * @throws Exception With invalid data source name.
     */
    protected static function openFile(string $dirLocation, string $dataSourceName) : array {
        $pathFile = realpath($dirLocation).DIRECTORY_SEPARATOR.$dataSourceName.'.json';

        if (!file_exists($pathFile)) {
            throw Exception::invalidFileName($dataSourceName);
        }

        $json = file_get_contents($pathFile);
        $dataSourceConfig = json_decode($json, true);
        if(!isset($dataSourceConfig[$dataSourceName])) {
            throw Exception::invalidDataSourceName($dataSourceName);
        }

        return $dataSourceConfig[$dataSourceName];
    }

    /**
     * @param string $dirLocation
     * @param string $dataSourceName
     *
     * @return Configuration
     *
     * @throws Exception With invalid data source config file.
     */
    public static function factoryFromFileLocation(string $dirLocation, string $dataSourceName) : Configuration {
        $dataSourceConfig = static::openFile($dirLocation, $dataSourceName);
        return static::factoryFromJsonData($dataSourceConfig, $dataSourceName);
    }

    /**
     * Returns a Configuration from an data source configuration array and a data source name.
     *
     * @param $dataSourceConfig
     * @param $dataSourceName
     *
     * @return Configuration
     *
     * @throws Exception
     */
    protected static function factoryFromJsonData(array $dataSourceConfig, string $dataSourceName) : Configuration {
        $columns = $dataSourceConfig['columns'];

        $instance = new static($dataSourceName, $columns);
        $instance->groupBy = $dataSourceConfig['groupBy'] ?? $instance->groupBy;
        if(!empty($instance->groupBy)){
            $instance->columns = $instance->groupBy;
        }

        if (isset($dataSourceConfig['tableName'])) {
            $instance->tableName = $dataSourceConfig['tableName'];
            if (isset($dataSourceConfig['primaryKeys'])) {
                $instance->setPrimaryKeyColumns($dataSourceConfig['primaryKeys']);
                if (isset($dataSourceConfig['sequentialColumn'])) {
                    $instance->setSequentialColumn($dataSourceConfig['sequentialColumn']);
                }
            }

            $instance->relations = $dataSourceConfig['relations'] ?? $instance->relations;
        }

        $instance->query = $dataSourceConfig['query'] ?? $instance->query;
        $instance->orderBy = $dataSourceConfig['orderBy'] ?? $instance->orderBy;
        $instance->resultSetLimit = $dataSourceConfig['resultSetLimit'] ?? $instance->resultSetLimit;
        $instance->conditions = $dataSourceConfig['conditions'] ?? $instance->conditions;
        $instance->parameters = $dataSourceConfig['parameters'] ?? $instance->parameters;
        $instance->parameters = array_combine($instance->parameters, $instance->parameters);
        $instance->defaults = $dataSourceConfig['defaults'] ?? $instance->defaults;

        if (isset($dataSourceConfig['columnType'])) {
            $instance->columnType = $dataSourceConfig['columnType'];
        }

        $instance->metaData = $dataSourceConfig;

        return $instance;
    }

    /**
     * Returns a Configuration from a relation metadata's array.
     *
     * @param array $relationMetaData The relation metadata.
     *
     * @return Configuration
     *
     * @throws Exception
     */
    public static function factoryFromRelation(array $relationMetaData) : Configuration {
        $relationConfig = new static($relationMetaData['targetTable'], $relationMetaData['targetColumns']);
        $relationConfig->tableName = $relationMetaData['targetTable'];
        $relationConfig->setPrimaryKeyColumns($relationMetaData['targetColumns']);
        $relationConfig->setSequentialColumn($relationMetaData['targetSequentialColumn']);
        return $relationConfig;
    }

    public function getName() : string {
        return $this->name;
    }

    public function getDriverName() : string {
        return $this->driverName;
    }

    public function setDriverName(string $driverName) {
        $this->driverName = $driverName;
    }

    public function validateDriverName($arrayDbName)
    {
        return $this->getDriverName() && in_array($this->getDriverName(), $arrayDbName);
    }

    public function getMetaData() : string {
        return $this->metaData;
    }

    /**
     * @return string[]
     */
    public function getColumns()
    {
        return $this->columns;
    }

    public function getPrimaryKeyColumns() : array {
        return $this->primaryKeyColumns;
    }

    public function getRelations() : array {
        return $this->relations;
    }

    public function getSequentialColumn() {
        return $this->sequentialColumn;
    }

    public function getTableName() {
        return $this->tableName;
    }

    public function hasQuery() : bool {
        return $this->query !== '';
    }
    
    public function hasGroupby() : bool {
        return !empty($this->groupBy);
    }

    /**
     * @return string
     */
    public function getQuery() {
        $query = $this->query;
        if($this->driverName && !empty($this->metaData['query_'.$this->driverName])){
            $query = $this->metaData['query_'.$this->driverName];
        }
        return $query;

    }

    /**
     * Returns the columns for select. Data columns if query, columns by default.
     *
     * @return array
     */
    public function getColumnsForSelect() : array {
        $columns = $this->hasQuery()
            ? $this->getDataColumns()
            : $this->getColumns();
        $groupBy = $this->getGroupBy();
        $hasGroupBy = $this->hasGroupby();
        /* To maintain the return pattern to the frontend */
        if ($this->driverName && in_array($this->driverName, QuerySelector::DRIVER_AS_IN_QUERY)) {
            $columnsAs = array();
            if(!$hasGroupBy){
                foreach ($columns as $column) {
                    $columnUpper = strtoupper($column);
                    $columnsAs[] = <<<SQL
                    $column AS "$columnUpper"
SQL;
                }
                
            }else{
                foreach ($this->groupBy as $column) {
                    $columnUpper = strtoupper($column);
                    $columnsAs[] = <<<SQL
                    $column AS "$columnUpper"
SQL;
                }
            }
        } else {
            $columnsAs = $columns;
        }
        return $columnsAs;
    }

    /**
     * @param string $dataColumn The data column name.
     *
     * @throws Exception Data column not found.
     *
     * @return string The column name.
     */
    public function getColumnByDataColumn(string $dataColumn) : string {
        if (isset($this->columns[$dataColumn])) {
            return $this->columns[$dataColumn];
        }

        throw Exception::dataColumnDoesNotExist($dataColumn, $this->name);
    }

    /**
     * Returns the data column by column name.
     *
     * @return string
     */
    public function getDataColumnByColumn(string $column) : string {
        if (isset($this->dataColumnsByColumn[$column])) {
            return $this->dataColumnsByColumn[$column];
        }

        throw Exception::dataColumnDoesNotExist($column, $this->name);
    }

    /**
     * Returns the data columns.
     *
     * @return array
     */
    public function getDataColumns() : array {
        return array_keys($this->columns);
    }

    /**
     * Sets the data source columns.
     *
     * @param array $columns
     */
    protected function setColumns(array $columns) {
        $this->dataColumnsByColumn = $this->columns = [];
        foreach ($columns as $dataColumnName => $columnName) {
            if (is_numeric($dataColumnName)) { // This allow BC when dataColumnName became necessary!
                $dataColumnName = $columnName;
            }
            $this->columns[$dataColumnName] = $columnName;
            $this->dataColumnsByColumn[$columnName] = $dataColumnName;
        }
    }

    /**
     * Returns the columns for result set. Data columns if query, columns by default.
     *
     * @return array
     */
    public function getColumnsForResultSet($onlyName = false) : array {
        $columns = $this->hasQuery()
            ? $this->getDataColumns()
            : $this->getColumns();
        /* To maintain the return pattern to the frontend */
        if (!$onlyName && $this->driverName && in_array($this->driverName, QuerySelector::DRIVER_AS_IN_QUERY)) {
            $columnsAs = array();
            if(!$this->hasGroupby()){
                foreach ($columns as $column) {
                    $columnUpper = strtoupper($column);
                    $columnsAs[] = <<<SQL
                    $column AS "$columnUpper"
SQL;
                }
                
            }
        } else {
            $columnsAs = $columns;
        }
        return $columnsAs;
    }

    public function getOrderBy() : array {
        return $this->orderBy;
    }

    public function getGroupBy() : array {
        return $this->groupBy;
    }

    public function getResultSetLimit() : int {
        return $this->resultSetLimit;
    }

    public function getConditions() : array {
        return $this->conditions;
    }

    public function getParameters() : array {
        return $this->parameters;
    }

    public function getDefaults() : array {
        return $this->defaults;
    }
}