<?php
namespace Zeedhi\Preferences\API\Controller\Crud;

use Zeedhi\Preferences\API\Controller\PreferenceException;

use Zeedhi\Framework\DTO;
use Zeedhi\Framework\DTO\Response\Notification;
use Zeedhi\Framework\Controller\Crud;
use Zeedhi\Framework\DataSource\DataSet;
use Zeedhi\Framework\DataSource\Manager;
use Zeedhi\Framework\DataSource\Manager\Exception as ManagerException;

class PreferencesCrud extends Crud {

    protected $parameterBag;
    protected $environment;

    public function __construct(Manager $dataSourceManager, $parameterBag, $environment){
        $this->parameterBag = $parameterBag;
        $this->environment = $environment;
        parent::__construct($dataSourceManager);
    }

    private function renameLabel($rows) {
        if(isset($rows[0]) && isset($rows[0]['LABEL'])){
            foreach($rows as &$row){
                $row['LABEL_TO_PREFERENCE'] = $row['LABEL'];
                unset($row['LABEL']);
            }
        }
        return $rows;
    }

    public function find(DTO\Request\Filter $request, DTO\Response $response){
        $this->parameterBag->set('ORGANIZATION_ID', $this->environment->getOrganizationId());
        $this->parameterBag->set('USER_ID', $this->environment->getUserId());
        $filterData = $request->getFilterCriteria();
        $filterData->setDataSourceName($this->dataSourceName);
        $dataSet = $this->dataSourceManager->findBy($filterData);
        $rows = $this->renameLabel($dataSet->getRows());
        $name = $dataSet->getDataSourceName();
        $newDataSet = new DataSet($name, $rows);
        $response->addDataSet($newDataSet);
    }

    protected function generateUniqueId(){
        return sha1(uniqid(rand()));
    }

    protected function prepareRowId($row){
        if(!isset($row['ID']) || !isset($row['__generate_id']) || $row['__generate_id'] !== false){
            $row['ID'] = $this->generateUniqueId();
        }
    }

    protected function validateDataSetUserId(DTO\Request\DataSet $request){
        $rows = $request->getDataSet()->getRows();
        $userId = $this->environment->getUserId();
        foreach ($rows as $row){
            if(isset($row['LABEL_TO_PREFERENCE'])){
                $row["LABEL"] = $row["LABEL_TO_PREFERENCE"];
                unset($row["LABEL_TO_PREFERENCE"]);
            }
            if (isset($row['__is_new']) && $row['__is_new'] === true) {
                $row['USER_ID'] = $userId;
                $this->prepareRowId($row);
            } else if($row['USER_ID'] !== $userId){
                throw PreferenceException::invalidDataSetUserId();
            }
        }
    }

    public function save(DTO\Request\DataSet $request, DTO\Response $response){
        $this->validateDataSetUserId($request);
        parent::save($request, $response);
    }

    protected function crudDelete(DTO\Request\DataSet $request, DTO\Response $response) {
        $dataSet = $request->getDataSet();
        $dataSet->setDataSourceName($this->dataSourceName);
        $deletedRows = $this->dataSourceManager->delete($dataSet);
        $response->addDataSet(new DataSet($this->dataSourceName, $deletedRows));
        $nrDeletedRows = count($deletedRows);
        $response->addNotification(new Notification("{$nrDeletedRows} row(s) deleted.", Notification::TYPE_SUCCESS));
    }

    public function delete(DTO\Request\DataSet $request, DTO\Response $response){
        try {
            $this->validateDataSetUserId($request);
            $this->crudDelete($request, $response);
        } catch (ManagerException $e) {
            $errorCode = $this->getErrorCode($e->getMessage());

            $response->setError(new DTO\Response\Error(
                $errorCode ?: $e->getMessage(),
                $e->getCode(),
                $e->getTraceAsString())
            );
        } catch (\Exception $e) {
            $response->setError(new DTO\Response\Error(
                $e->getMessage(),
                $e->getCode(),
                $e->getTraceAsString())
            );
        }
    }

    protected function getORAErrorCode($errorMessage) {
        preg_match('#ORA-[0-9]+#', $errorMessage, $matches);

        if (isset($matches[0])) {
            return $matches[0];
        } else {
            return null;
        }
    }

    protected function getErrorCode($error) {
        $oraErrorCode = $this->getORAErrorCode($error);

        switch ($oraErrorCode) {
            case 'ORA-02292': return 'delete_integrity_error';
            default:          return $error;
        }
    }

}
