<?php
namespace tests\Zeedhi\Preferences\API\Controller\Crud;

use Zeedhi\Framework\DTO;
use Zeedhi\Framework\DataSource\Manager;
use Zeedhi\Framework\DataSource\DataSet;
use Zeedhi\Framework\DataSource\ParameterBag;
use Zeedhi\Framework\DataSource\FilterCriteria;

use Zeedhi\Preferences\API\Helper\Environment;
use Zeedhi\Preferences\API\Controller\PreferenceException;
use Zeedhi\Preferences\API\Controller\Crud\PreferencesCrud;

class PreferencesCrudTest extends \PHPUnit_Framework_TestCase {

    /** @var Manager|\PHPUnit_Framework_MockObject_MockObject */
    private $dataSourceManager;
    /** @var ParameterBag|\PHPUnit_Framework_MockObject_MockObject */
    private $parameterBag;
    /** @var Environment|\PHPUnit_Framework_MockObject_MockObject */
    private $environment;
    /** @var PreferencesCrud */
    private $preferencesCrud;
    /** @var PreferenceException */
    private $preferenceException;

    protected function setUp() {
        $this->dataSourceManager = $this->getMockForAbstractClass(Manager::class);

        $this->preferenceException = $this->getMockBuilder(PreferenceException::class)
            ->disableOriginalConstructor()
            ->setMethods(array('invalidDataSetUserId'))
            ->getMock();

        $this->parameterBag = $this->getMockBuilder(ParameterBag::class)
            ->disableOriginalConstructor()
            ->setMethods(array('set'))
            ->getMock();

        $this->environment = $this->getMockForAbstractClass(Environment::class);

        $this->preferencesCrud = new PreferencesCrud($this->dataSourceManager, $this->parameterBag, $this->environment);
    }

    public function testFind(){
        $request = new DTO\Request\Filter(new FilterCriteria(''), 'POST', '/preference', 'userId');
        $response = new DTO\Response();
        $dataSet = new DataSet('dataSetName', array());

        $this->dataSourceManager->expects($this->once())
            ->method('findBy')
            ->willReturn($dataSet);

        $this->environment->expects($this->once())
            ->method('getOrganizationId')
            ->willReturn(1);

        $this->environment->expects($this->once())
            ->method('getUserId')
            ->willReturn('userId');

        $this->preferencesCrud->find($request, $response);
        $this->assertEquals('dataSetName', $response->getDataSets()[0]->getDataSourceName());
    }

    public function testSave(){
        $rows = array(
            array(
                "__is_new" => false,
                "ID"       => 1,
                "USER_ID"  => "userId",
                "LABEL_TO_PREFERENCE" => "First"
            ),
            array(
                "__is_new" => true,
                "ID"       => 2,
                "USER_ID"  => "userId",
                "LABEL_TO_PREFERENCE" =>"Second"
            )
        );
        $dataSet = new DataSet("filter", $rows);

        $request = new DTO\Request\DataSet($dataSet, 'POST', '/preference', 'userId');
        $response = new DTO\Response();

        $this->dataSourceManager->expects($this->once())
            ->method('persist')
            ->with($dataSet)
            ->willReturn(array(0,1,2));

        $this->dataSourceManager->expects($this->once())
            ->method('populateDataSet')
            ->with($dataSet)
            ->willReturn($dataSet);

        $this->environment->expects($this->once())
            ->method('getUserId')
            ->willReturn('userId');

        $this->preferencesCrud->save($request, $response);
        $notifications = $response->getNotifications()[0];
        $this->assertEquals('success', $notifications->getType());
        $this->assertEquals('3 row(s) persisted with success', $notifications->getMessage());
    }

    public function testDelete(){
        $rows = array(
            array(
                "__is_new" => false,
                "ID"       => 1,
                "USER_ID"  => "userId",
                "LABEL_TO_PREFERENCE" => "First"
            ),
            array(
                "__is_new" => false,
                "ID"       => 2,
                "USER_ID"  => "userId",
                "LABEL_TO_PREFERENCE" =>"Second"
            )
        );
        $dataSet = new DataSet("filter", $rows);

        $request = new DTO\Request\DataSet($dataSet, 'POST', '/preference', 'userId');
        $response = new DTO\Response();

        $this->dataSourceManager->expects($this->once())
            ->method('delete')
            ->with($dataSet)
            ->willReturn(array(0,1,2));

        $this->environment->expects($this->once())
            ->method('getUserId')
            ->willReturn('userId');

        $this->preferencesCrud->delete($request, $response);
        $notifications = $response->getNotifications()[0];
        $this->assertEquals('success', $notifications->getType());
        $this->assertEquals('3 row(s) deleted.', $notifications->getMessage());
    }

    public function testSaveThrowsExceptionWhenUserIsInvalid(){
        $row = array(
            "__is_new" => false,
            "USER_ID"  => 'otherUser',
            "LABEL_TO_PREFERENCE" => "First"
        );
        $dataSet = new DataSet("filter", array($row));

        $request = new DTO\Request\DataSet($dataSet, 'POST', '/preference', 'userId');
        $response = new DTO\Response();

        $this->environment->expects($this->once())
            ->method('getUserId')
            ->willReturn('userId');

        $expectedException = PreferenceException::invalidDataSetUserId();
        $this->expectException(PreferenceException::class);
        $this->expectExceptionMessage($expectedException->getMessage());
        $this->expectExceptionCode($expectedException->getCode());

        $this->preferencesCrud->save($request, $response);
    }

    public function testDeleteThrowsExceptionWhenUserIsInvalid(){
        $row = array(
            "__is_new" => false,
            "USER_ID"  => 'otherUser',
            "LABEL_TO_PREFERENCE" =>"Second"
        );
        $dataSet = new DataSet("filter", array($row));
        $request = new DTO\Request\DataSet($dataSet, 'POST', '/preference', 'userId');
        $response = new DTO\Response();
        $expectedException = PreferenceException::invalidDataSetUserId();

        $this->preferencesCrud->delete($request, $response);
        $this->assertEquals($response->getError()->getErrorCode(), $expectedException->getCode());
        $this->assertEquals($response->getError()->getMessage(), $expectedException->getMessage());
    }
}
