<?php
namespace Zeedhi\Report\ReportStrategy;

/**
 * Generates a pdf report.
 */
class PDF extends \FPDF implements ReportStrategy {

    /** @const number CM_TO_PX_BASE Value base to convert centimeters to pixels.  */
    const CM_TO_PX_BASE = 37.7952755906;
    /** @const number TAB_SIZE Tab size.  */
    const TAB_SIZE = 4;
    private $isWebService; // Variable to indicate whether it's a web service or not

    /**
     * Returns the height of the footer.
     *
     * @param boolean $isWebService Indicates whether it's a web service.
     * @return int Footer height.
     */
    public static function FOOTER_HEIGHT($isWebService) {
        return $isWebService ? 16 : 5; // Returns 16 if it's a web service, otherwise returns 5
    }

    /**
     * Returns the blank space needed to prevent the totalizer from being printed on the footer.
     *
     * @param boolean $isWebService Indicates whether it's a web service.
     * @return int Blank space needed.
     */
    public static function BLANK_SPACE($isWebService) {
        return $isWebService ? 8 : 4; // Returns 8 if it's a web service, otherwise returns 4
    }

    private $CellHeightBasedOnTitle;

    private $thisLastPage = false;
    /** @var int $totalValues Defines the actual global total value of each column. */
    protected $totalValues = [];
    /** @var array $columns Report columns. */
    protected $columns;
    /** @var array $metaData Report metadata. */
    protected $metaData;
    /** @var array $filterSize The amount of rows used to print the filter info. */
    protected $filterSize;
    /** @var int $textSize The height of the text. */
    protected $textSize = 4;
    /** @var bool $hasExpression Defines if there is expressions on the report.  */
    protected $hasExpression;
    /** @var array $groupValues The value of the groups. */
    protected $groupValues = [];
    /** @var int $indentLevel Defines the actual indent level. */
    protected $indentLevel;
    /** @var array $groups The report groups. */
    protected $groups;
    /** @var array $groupNames The report group names. */
    protected $groupNames;
    /** @var array $momentToPhpDateDictionary Dictionary to translate the moment format. */
    protected $momentToPhpDateDictionary = [
        'M'     => 'n',
        'Mo'    => '',
        'MM'    => 'm',
        'MMM'   => 'M',
        'MMMM'  => 'F',
        'Q'     => '',
        'Qo'    => '',
        'D'     => 'j',
        'Do'    => 'jS',
        'DD'    => 'd',
        'DDD'   => 'z',
        'DDDo'  => '',
        'DDDD'  => '',
        'd'     => 'w',
        'do'    => '',
        'dd'    => 'd',
        'ddd'   => '',
        'dddd'  => '' ,
        'e'     => 'w',
        'E'     => '',
        'w'     => '',
        'wo'    => '',
        'ww'    => '',
        'W'     => '',
        'Wo'    => '',
        'WW'    => '',
        'YY'    => 'y',
        'YYYY'  => 'Y',
        'Y'     => 'Y',
        'yyyy'  => 'Y',
        'yy'    => 'y',
        'gg'    => '',
        'gggg'  => '',
        'GG'    => '',
        'GGGG'  => '',
        'A'     => 'A',
        'a'     => 'a',
        'H'     => 'G',
        'HH'    => 'H',
        'h'     => 'g',
        'hh'    => 'h',
        'k'     => '',
        'kk'    => '',
        'm'     => '',
        'mm'    => 'm',
        's'     => '',
        'ss'    => 's',
        'S'     => '',
        'z'     => 'T',
        'zz'    => 'T',
        'Z'     => 'P',
        'ZZ'    => 'O',
        'X'     => 'U',
        'x'     => ''
    ];
    /** @var array $langMask Defines the default values for the language mask. */
    protected $langMask = [
        'en_us' => [
            'symbol' => 'US$',
            'thousands' => ',',
            'decimal' => '.'
        ],
        'pt' => [
            'symbol' => '€',
            'thousands' => ',',
            'decimal' => '.'
        ],
        'pt_br' => [
            'symbol' => 'R$',
            'thousands' => '.',
            'decimal' => ','
        ],
        'es' => [
            'symbol' => '$',
            'thousands' => '.',
            'decimal' => ','
        ],
        'es_co' => [
            'symbol' => 'CO$',
            'thousands' => '.',
            'decimal' => ','
        ],
        'es_cl' => [
            'symbol' => '',
            'thousands' => '.',
            'decimal' => ','
        ]
    ];
    /**
     * Language dictionary for translations.
     *
     * Contains translations  in different languages.
     */
    protected $languageDictionary = [
        'en_us' => [
            'filter_used' => 'Filters Used',
            'no_filter' => 'No filters were provided'
        ],
        'pt_br' => [
            'filter_used' => 'Filtros Utilizados',
            'no_filter' => 'Nenhum filtro foi informado'
        ],
        'es_cl' => [
            'filter_used' => 'Filtros Utilizados',
            'no_filter' => 'Ningún filtro ha sido informado'
        ]
    ];

    /** @var array $defaultMaskValues The default mask values. */
    protected $defaultMaskValues = [
        "showSymbol" => true,
        "precision" => 2,
        "allowZero" => true,
        "allowNegative" => true
    ];
    /** @var array $masks The masks and it's params. */
    protected $masks;
    /** @var array $fontConfig Dictionary for the font types used on the report generation. */
    protected $fontConfig = [
        "family"=>"Arial",
        "title"=>[
            "style"=> "B",
            "size"=> 16
        ],
        "bCommon"=>[
            "style"=> "B",
            "size"=> 8
        ],
        "common"=>[
            "style"=> "",
            "size"=> 8
        ],
        "footer"=>[
            "style"=>"",
            "size"=> 6
        ],
        "bCommonGreater"=>[
            "style"=> "B",
            "size"=> 12
        ],
        "commonGreater"=>[
            "style"=> "",
            "size"=> 11
        ]
    ];

    /**
     * Constructor.
     *
     * @param array $metaData Report metadata.
     */
    public function __construct($metaData, $isWebService = false){
        $orientation = strtoupper(substr($metaData['orientation'], 0, 1));
        parent::__construct($orientation, 'mm', 'A4');
        $this->metaData = $metaData;
        $this->isWebService = $isWebService;
        $this->groups = $this->configGroups($metaData['groups']);
        $this->columns = $this->configColumns($metaData['columns']);
        $this->masks = $this->configMasks($metaData['lang']);
        $this->currentGroup = 0;
        $this->SetAutoPageBreak(false, self::FOOTER_HEIGHT($this->isWebService));
        $this->start();
    }

    /**
     * Retrieves the translation for the given language.
     *
     * If translation for the specified language is not available, falls back to Portuguese (Brazil).
     *
     * @param string $inputLang The language code for the requested translation. Default is 'pt_br'.
     * @return string The translated phrase.
     */
    private function getTranslations($key, $inputLang = 'pt_br'){
        return isset($this->languageDictionary[$inputLang][$key]) ? $this->languageDictionary[$inputLang][$key] : $this->languageDictionary['pt_br'][$key];
    }

    /**
     * Config the masks.
     *
     * Builds the array with each default params of the masks.
     *
     * @param string $inputLang The language used to generate the report.
     * @return array The array with each default params of the masks.
     */
    private function configMasks($inputLang){
        $this->defaultMaskValues['inputLang'] = $inputLang;
        return [
            "number" => [
                "allowZero" => $this->defaultMaskValues["allowZero"],
                "allowNegative" => $this->defaultMaskValues['allowNegative']
            ],
            "float" => [
                "thousands" => $this->langMask[$this->defaultMaskValues['inputLang']]['thousands'],
                "decimal" => $this->langMask[$this->defaultMaskValues['inputLang']]['decimal'],
                "precision" => $this->defaultMaskValues['precision'],
                "allowZero" => $this->defaultMaskValues["allowZero"],
                "allowNegative" => $this->defaultMaskValues['allowNegative']
            ],
            "viewFloat" => [
                "thousands" => $this->langMask[$this->defaultMaskValues['inputLang']]['thousands'],
                "decimal" => $this->langMask[$this->defaultMaskValues['inputLang']]['decimal'],
                "precision" => $this->defaultMaskValues['precision'],
                "allowZero" => $this->defaultMaskValues["allowZero"],
                "allowNegative" => $this->defaultMaskValues['allowNegative']
            ],
            "currency" => [
                "showSymbol" => $this->defaultMaskValues['showSymbol'],
                "symbol" => $this->langMask[$this->defaultMaskValues['inputLang']]['symbol'],
                "thousands" => $this->langMask[$this->defaultMaskValues['inputLang']]['thousands'],
                "decimal" => $this->langMask[$this->defaultMaskValues['inputLang']]['decimal'],
                "precision" => $this->defaultMaskValues['precision'],
                "allowZero" => $this->defaultMaskValues["allowZero"],
                "allowNegative" => $this->defaultMaskValues['allowNegative']
            ],
            "percent" => [
                "minValue" => 0,
                "maxValue" => 100,
                "precision" => $this->defaultMaskValues['precision'],
                "allowZero" => $this->defaultMaskValues["allowZero"],
                "allowNegative" => $this->defaultMaskValues['allowNegative'],
                "thousands" => $this->langMask[$this->defaultMaskValues['inputLang']]['thousands'],
                "decimal" => $this->langMask[$this->defaultMaskValues['inputLang']]['decimal']
            ],
            "date" => [
                "format" => "dd/mm/yyyy"
            ],
            "time" => [
                "withSeconds" => false
            ],
            "datetime" => [
                "format" => "dd/mm/yyyy"
            ],
            "fix" => [
                "mask" => ""
            ],
            "zerofill" => [
                "maxlength" => 2
            ]
        ];
    }

    /**
     * Start the report.
     *
     * Sets the margin values and add a page.
     */
    public function start(){
        $this->SetMargins(10, 5);
        $this->AddPage();
    }

    /**
     * Return the column size.
     *
     * Calculate the size of the column in pixels.
     *
     * @param string $size The column size.
     * @return float The column size in pixels.
     */
    private function getColumnSize($size) {
        if (isset($size)) {
            if (substr($size, -1) === "%") {
                $tabSize = isset($this->metaData["columns"]) ? count($this->metaData["groups"]) * 4 : 0;
                $size = intval($size);
                $margin = $this->lMargin + $this->rMargin;
                $pageWidth = ($this->w - $margin) - $tabSize;
                $columnSize = ($pageWidth / 100) * $size;
            } else {
                $columnSize = $size;
            }
        } else {
            $columnSize = 20 * self::CM_TO_PX_BASE;
       }

        return $columnSize;
    }

    /**
     * Config the value of the report groups.
     *
     * Sets the groups values and the group names.
     *
     * @param array @groups The metadata groups to set the global groups.
     * @return array The groups on the report format.
     */
    private function configGroups($groups){
        $builtGroups = [];

        foreach($groups as $group){
            $builtGroups[$group['name']] = $group;
            $this->groupValues[$group['name']]['value'] = "";
        }

        $this->groupNames = array_keys($builtGroups);
        return $builtGroups;
    }

    /**
     * Initialize the value of the expressions.
     *
     * Initialize the value of the expressions setting 'COUNT', 'MIN', 'MAX' and 'SUM' as
     * as, zero, PHP_INT_MAX, ~PHP_INT_MAX, zero respectively. For 'AVG' initialize the value of
     * 'COUNT' ans 'SUM'.
     *
     * @param string $columnName The column in which the expression is related.
     * @param string $column     The column metadata.
     */
    private function initializeExpressions($columnName, $column){
        if(!isset($column['expression'])) return;
        $this->hasExpression = true;
        foreach($this->groupValues as $key => $group){
            $this->groupValues[$key][$columnName] = [];
            switch($column['expression']){
                case 'COUNT':
                    $this->groupValues[$key][$columnName]['COUNT'] = 0;
                    break;
                case 'MIN':
                    $this->groupValues[$key][$columnName]['MIN'] = PHP_INT_MAX;
                    break;
                case 'MAX':
                    $this->groupValues[$key][$columnName]['MAX'] = ~PHP_INT_MAX;
                    break;
                case 'SUM':
                    $this->groupValues[$key][$columnName]['SUM'] = 0;
                    break;
                case 'AVG':
                    $this->groupValues[$key][$columnName]['COUNT'] = 0;
                    $this->groupValues[$key][$columnName]['SUM'] = 0;
                    break;
            }
        }
    }

    /**
     * Config columns with the expected format for the report generation.
     *
     * Config columns with the properties align, order and size with the expected format for the report.
     * Also initialize the group expressions.
     *
     * @param array $columns The columns with the configuration for the report.
     * @return array The columns with the expected format for the report.
     */
    private function configColumns($columns){
        foreach($columns as $key => $column){
            $column['align'] = strtoupper(substr($column['align'], 0, 1));
            $column['order'] = $column['sequence'];
            $column['size'] = $this->getColumnSize($column['size']);
            $this->initializeExpressions($key, $column);
            $columns[$key] = $column;
        }

        uasort($columns, function($colA, $colB){
            return ($colA['sequence'] <= $colB['sequence']) ? -1 : 1;
        });
        return $columns;
    }

    /**
     * Set the font based on a specific configuration.
     *
     * Uses the function SetFont to configure the font to be used. This is done based on an array
     * of fonts already configured.
     *
     * @param string $type The type of font to be configured. The values accepted are: 'title',
     * 'common' and 'bCommon'.
     */
    private function configFont($type='common'){
        $this->SetFont($this->fontConfig['family'],
            $this->fontConfig[$type]['style'],
            $this->fontConfig[$type]['size']);
    }

    /**
     * Calculate the height of a cell based on the length of its title.
     *
     * This function calculates the height of a cell in a document based on the length
     * of the title provided. If the title length exceeds 44 characters, a height of 5
     * units is returned; otherwise, a height of 15 units is returned.
     *
     * @param string $title The title of the cell.
     * @return int The height of the cell in units.
     */
    private function calculateCellHeight($title){
        $titleLength = strlen($title);
        return ($titleLength > 44) ? 5 : 15;
    }

    /**
     * Prints the title of the report.
     *
     * Prints a cell centered with the report name with the font 'title'.
     */
    private function printTitle($customTitle){
        $this->configFont('title');
        $title = $this->internationalize($customTitle);

        $leftMargin = 10;
        $rightMargin = $this->GetPageWidth() - $this->GetX() - 30;
        $availableWidth = $rightMargin - $leftMargin;

        $this->CellHeightBasedOnTitle = $this->calculateCellHeight($title);
        $this->MultiCell($availableWidth, $this->CellHeightBasedOnTitle, $title, 0, 'C', false);

        $this->SetX($rightMargin);
    }

    /**
     * Internationalize the given word.
     *
     * Encode the given word from 'UTF-8' to 'ISO-8859-1'.
     *
     * @param string $word Word to be internationalized.
     * @return string The word internationalized.
     */
    private function internationalize($word){
        return iconv("UTF-8", "ISO-8859-1//TRANSLIT", $word);
    }

    /**
     * Translates the given word.
     *
     * Maps the translated word using the words array.
     *
     * @param string $word Word to be translated.
     * @return string The word translated.
     */
    private function translate($word){
        $word = $this->metaData['words'][$word];
        $internationalized = $this->internationalize($word);
        return $internationalized;
    }

    /**
     * Print the current page on the report.
     *
     * Prints a cell with the current page number with the font 'common'.
     */
    private function printPageNum(){
        $this->configFont();
        $word = $this->translate('Page');
        $this->AliasNbPages('total_pages');
        $rightMargin = $this->GetPageWidth() - 10;
        $this->SetX($rightMargin);

        $cellHeight = ($this->CellHeightBasedOnTitle == 15) ? -20 : -10;
        $this->Cell(7, $cellHeight, 'Page '.$this->PageNo().'/total_pages', 0, 0, 'R');
        }

    /**
     * Print the report generation date.
     *
     * Print a cell with the report generation date with the font 'common' and the format 'd/m/Y H:i:s'.
     */
    private function printDate(){
        $date = date('d/m/Y H:i:s');
        $this->configFont();
        $cellHeight = ($this->CellHeightBasedOnTitle == 15) ? -25 : -15;
        $this->Cell(0, $cellHeight, $date, 0, 0, 'R');
    }

    /**
     * Print a line.
     *
     * Print a line parallel to the x axis. on the current y respecting the configured margins.
     */
    private function printLine(){
        $x1 = $this->lMargin;
        $x2 = $this->GetPageWidth() - $this->rMargin;
        $y = $this->GetY();

        $this->Line($x1, $y, $x2, $y);
    }

    /**
     * Tests if there will be overflow if the given text is printed on the screen.
     *
     * Tests if the current x plus the given string is bigger than the page width minus the right margin.
     *
     * @param string $str String to test the overflow.
     * @return bool The result of the overflow test.
     */
    private function testWidthOverflow($str){
        return $this->GetX() + $this->GetStringWidth($str) > $this->GetPageWidth() - $this->rMargin;
    }

    /**
     * Prints the rect to be the filter background.
     *
     * Prints a rect that fill the height of a filter row and the width of the page minus the margins.
     */
    private function printFilterRect(){
        $x = $this->lMargin;
        $y = $this->GetY();
        $w = $this->GetPageWidth() - ($this->lMargin * 2);
        $h= $this->textSize;

        $this->SetFillColor(255);
        $this->Rect($x, $y, $w, $h, 'F');
    }

    /**
     * Breaks the filter line.
     *
     * Breaks the filter line setting the correct x axis and print it's rect.
     */
    private function breakFilterLine(){
        $this->SetX(0 + $this->lMargin);
        $this->SetY($this->GetY() + ($this->textSize / 2));
        $this->printFilterRect();
        $this->SetY($this->GetY() + ($this->textSize / 2));
    }

    /**
     * Writes the filter value on the report.
     *
     * Writes the filter value word by word, testing if there will be a overflow and, breaking the
     * line if there will be.
     *
     * @param string $value The text to be printed on the report filter.
     */
    private function writeFilterValue($value){
        $this->configFont('commonGreater');
        $text = $this->internationalize($value);
        $text = explode(" ", $text);
        foreach($text as $word){
            $word .= " ";
            if($this->testWidthOverflow($word)){
                $this->breakFilterLine();
                $this->filterSize++;
            }
            $this->Write(null, $word);
        }
    }

    /**
     * Print the given filter.
     *
     * Print both label and value of the given filter.
     *
     * @param array $filter The filter to be printed.
     */
    private function printFilterText($filter){
        $this->filterSize++;
        $this->configFont('bCommonGreater');
        $this->Write(null, $this->internationalize($filter['label'].": "));
        $this->writeFilterValue($filter['value']);
    }

    /**
     * Print all filter used to generate the report.
     *
     * Test if there is a filter to be printed and, print it when there is. Also sets the filterSize var.
     */
    private function printFilter(){
        if(empty($this->metaData['filter'])) return;
        $this->SetY(21.5);
        $this->printLine();
        $this->breakFilterLine();

        $this->printLastSubTitle($this->getTranslations('filter_used', $this->metaData['lang']));

        foreach($this->metaData['filter'] as $filter){
            $this->breakFilterLine();
            $this->breakFilterLine();
            $this->printFilterText($filter);
        }
        $this->breakFilterLine();
    }

    private function areaHeader(){
        $this->filterSize = 0;
        $this->filterSize = 4 * $this->filterSize;
    }

    /**
     * Prints the given column with its name.
     *
     * Prints the given given column if it is not a group.
     *
     * @param string $col  The column name.
     * @param string $name The column description to be printed.
     */
    private function printColumnHeader($col, $name){
        if(!empty($this->groups[$col])) return;

        $name = $this->internationalize($name);
        $this->Cell($this->columns[$col]['size'], 3,
            $name, 0, 0,
            $this->columns[$col]['align'], true);
    }

    /**
     * Prints all columns header.
     *
     * Prints the header of all columns there are not groups.
     */
    private function printColumnsHeader(){
        if (!$this->thisLastPage) {
            $this->configFont('bCommon');
            $this->SetFillColor(255);
            $this->filterSize -= 1;

            $this->setY(21.5 + $this->filterSize);
            $this->printLine();
            $this->setY(22.5 + $this->filterSize);

            $this->SetX($this->GetX() + (sizeof($this->groups)) * self::TAB_SIZE);

                foreach($this->columns as $key => $column){
                    $this->printColumnHeader($key, $column['description']);
                }
            $this->setY(26.5 + $this->filterSize);
            $this->printLine();
            $this->setY(28 + $this->filterSize);
        }
    }

    /**
     * Print the image on the header.
     *
     * Print the image inside the header of the report.
     *
     * @param string $image  The image name.
     */
    private function printHeaderImage($image){
        if(isset($this->metaData[$image])){
            $pic = $this->metaData[$image];

            $dataPieces = explode(',', $pic);
            $type = explode( ';' ,explode('/', $dataPieces[0])[1])[0];

            $this->Cell(30, 10, "", 0, 0);
            $y = $this->GetY();
            $this->SetX(0 + $this->lMargin);
            $this->Image($pic, null, null, 30, 10, $type);
            $this->SetY($y);
        }
    }

    /**
     * Print the report header.
     *
     * Print the title, page number, date filter and the column's header.
     */
    public function Header(){
        $this->printHeaderImage('reportClientLogo');
        $this->printTitle($this->metaData['title']);
        $this->printPageNum();
        $this->printDate();
        $this->areaHeader();
        $this->printColumnsHeader();
    }

    /**
     * Print the group header of the given column with the given row value.
     *
     * Print the header of the group with the correct indentation testing if there is a label to
     * be added.
     *
     * @param array $row     The current row with the values for the header print.
     * @param string $column  The configuration for the group header print.
     */
    private function printGroupHeader($row, $column){
        $this->configFont('bCommon');
        $rowValue = "";

        $this->SetX($this->indentLevel * self::TAB_SIZE + $this->lMargin);

        if(is_null($column)) {
            if(isset($row['__groupLabel']) && strlen($row['__groupLabel']) > 0 ){
                $rowValue .= $row['__groupLabel'];
            }

            if(isset($row['__groupValue']) && strlen($row['__groupValue']) > 0 ){
                $rowValue .= strlen($rowValue) > 0 ?
                    ": ".$row['__groupValue'] : $row['__groupValue'];
            }
        } else {
            if(!empty($this->groups[$column]['header']['label'])){
                $rowValue .= $this->groups[$column]['header']['label'];
            }

            if(isset($row[$column]) && strlen($row[$column]) > 0){
                $value = $row[$column];
                $value = $this->maskPrintGroup($column,$value);
                $rowValue .= strlen($rowValue) > 0 ?
                    ": ".$value : $value;
            }
        }
        $rowValue = $this->internationalize($rowValue);
        $this->Cell(0, 5, $rowValue, 0, 0);
        $this->Ln();
    }

    /**
     * Reset the expression's value of the given group.
     *
     * Reset the value of the given group with the same values of the initialize.
     * @param string $group The group name to be reset.
     */
    private function resetGroupExpressions($group){
        foreach($this->groupValues[$group] as $columnName => $expressions){
            if(strcmp($columnName, 'value') != 0){
                foreach($expressions as $expression => $value){
                    switch($expression){
                        case 'COUNT':
                            $this->groupValues[$group][$columnName]['COUNT'] = 0;
                            break;
                        case 'MIN':
                            $this->groupValues[$group][$columnName]['MIN'] = PHP_INT_MAX;
                            break;
                        case 'MAX':
                            $this->groupValues[$group][$columnName]['MAX'] = ~PHP_INT_MAX;
                            break;
                        case 'SUM':
                            $this->groupValues[$group][$columnName]['SUM'] = 0;
                            break;
                        case 'AVG':
                            $this->groupValues[$group][$columnName]['COUNT'] = 0;
                            $this->groupValues[$group][$columnName]['SUM'] = 0;
                            break;
                    }
                }
            }
        }
    }

    /**
     * Print the value of given column's children groups and clear it.
     *
     * Print the value of given column's children groups and it's value and
     * expression value.
     *
     * @param string $currColumn The actual column name.
     */
    private function clearOncomingGroups($currColumn, $valueIsFormatted=false){
        if(empty($this->groupNames)) return;

        $count = count($this->groupNames);
        $groupPos = array_search($currColumn, $this->groupNames);

        for($i = $count - 1; $groupPos >=0 && $i >= $groupPos; $i--){
            $this->printGroupFooter($this->groupNames[$i], $valueIsFormatted);
            $this->resetGroupExpressions($this->groupNames[$i]);
            unset($this->groupValues[$this->groupNames[$i]]['value']);
        }
    }

    /**
     * Recalculates the group values considering the given row value.
     *
     * Recalculates the group values considering the given row value, adding one,
     * adding the row value, testing if the value is smaller, testing if the value
     * is bigger when the group's expression is 'COUNT', 'SUM', 'MIN', 'MAX', respectively.
     *
     * @param array $row Row to be processed.
     */
    private function processGroupValues($row){
        foreach($this->columns as $columnName => $column){
            foreach($this->groupValues as $group => $value){
                if(isset($column['expression'])){
                    switch($column['expression']){
                        case 'COUNT':
                            $this->groupValues[$group][$columnName]['COUNT']++;
                            break;
                        case 'SUM':
                            $this->groupValues[$group][$columnName]['SUM'] += $row[$columnName];
                            break;
                        case 'MIN':
                            if($row[$columnName] < $this->groupValues[$group][$columnName]['MIN'])
                                $this->groupValues[$group][$columnName]['MIN'] = $row[$columnName];
                            break;
                        case 'MAX':
                            if($row[$columnName] > $this->groupValues[$group]['MIN'])
                                $this->groupValues[$group][$columnName]['MIN'] = $row[$columnName];
                            break;
                        case 'AVG':
                            $this->groupValues[$group][$columnName]['COUNT'] += $row[$columnName];
                            $this->groupValues[$group][$columnName]['SUM'] += $row[$columnName];
                            break;
                    }
                }
            }
        }
    }

    /**
     * Print the groups of the current row.
     *
     * Tests if the value of the row is equal of the already printed and update the group values.
     *
     * @param array $row The row to have it's groups printed.
     * @param bool $rePrintGroups When it's true ignore if the rows of the groups are already printed and
     * print then all.
     */
    private function printGroups(array $row, $rePrintGroups = false, $valueIsFormatted=false){
        if(empty($this->groups)) return;

        $this->configFont('bCommon');
        $this->indentLevel = 0;

        if($rePrintGroups){
            foreach($this->groupValues as $key => $group){
                $this->printGroupHeader($row, $key);
                $this->groupValues[$key]['value'] = $row[$key];
                $this->indentLevel++;
            }
        }
        else {
            foreach($this->groupValues as $key => $group){
                if(empty($this->groupValues[$key]['value'])){
                    $this->printGroupHeader($row, $key);
                    $this->groupValues[$key]['value'] = $row[$key];
                    $this->indentLevel++;
                }
                elseif(strcmp($this->groupValues[$key]['value'], $row[$key]) != 0){
                    $this->clearOncomingGroups($key, $valueIsFormatted);
                    $this->groupValues[$key]['value'] = $row[$key];
                    $this->indentLevel = array_search($key, $this->groupNames);
                    $this->printGroupHeader($row, $key);
                    $this->indentLevel++;
                }
            }
        }

        $this->processGroupValues($row);
    }

    /**
     * Returns the group value of the given column name.
     *
     * Returns the given group value of the given column returning the count divided by the sum when
     * the expression is avg or the value stored when is another expression.
     *
     * @param string $columnName The column to have it's value calculated.
     * @param array $column      The column configuration with it's expression.
     * @param string $group      The name of the group that will have it's value returned.
     *
     * @return float The expression value.
     */
    private function getGroupExpressionResult($columnName, $column, $group){
        if(!isset($column['expression'])) return "";
        elseif($column['expression'] === 'AVG'){
            return $this->groupValues[$group][$columnName]['SUM'] /
                     $this->groupValues[$group][$columnName]['COUNT'];
        }
        else{
            return $this->groupValues[$group][$columnName][$column['expression']];
        }

    }
    private function maskPrintGroup($currentColumn,$value){
        $mask = null;
        $columnMetadata = $this->columns[$currentColumn];
        if(isset($columnMetadata['format']) && isset($columnMetadata['format']['type'])){
            $mask = $columnMetadata['format']['type'];
        }
        if($mask){
            $value = $this->formatValue($value, $columnMetadata);
        }
        return $value;
    }
    /**
     * Prints the footer of the given column.
     *
     * Prints the footer of the given column with the font 'bCommon'.
     *
     * @param string $currentColumn Column name to has it's footer printed.
     */
    private function printGroupFooter($currentColumn, $valueIsFormatted=false){
        $this->configFont('bCommon');
        $rowValue = "Total - ";
        if(!empty($this->groups[$currentColumn]['header']['label'])){
            $rowValue .= $this->groups[$currentColumn]['header']['label'].": ";
        }
        $this->SetX(array_search($currentColumn, $this->groupNames) * self::TAB_SIZE + $this->lMargin);
        $value = $this->groupValues[$currentColumn]['value'];
        $value = $this->maskPrintGroup($currentColumn,$value);
        $rowValue .= $value;
        $rowValue = $this->internationalize($rowValue);
        $this->Cell(4, 5, $rowValue, 0, 0);
        $this->Ln();

        $this->SetX(count($this->groups) * self::TAB_SIZE + $this->lMargin);
        foreach($this->columns as $columnName => $column){
            $value = $this->getGroupExpressionResult($columnName, $column, $currentColumn);
            $expression = null;

            if(isset($column["expression"])){
                $expression = $column['expression'];

                if(!isset($this->totalValues[$columnName][$currentColumn])){
                    $this->totalValues[$columnName][$currentColumn] = 0;
                }

                if(is_numeric($value)){
                    $this->totalValues[$columnName][$currentColumn] += $value;
                }
            }

            $mask = null;
            if(isset($column['format']) && isset($column['format']['type'])){
                $mask = $column['format']['type'];
            }

            if($this->isToFormat($valueIsFormatted, $column, $expression, $mask)){
               if(!empty($expression)) {
                   $value = $this->formatValue($value, $column);
               }
            }

            if($column['size'] != 0 && !in_array($columnName, $this->groupNames)) {
                $this->Cell(
                    $column['size'],
                    5, $value, 0, 0,
                    $column['align']
                );
            }
        }
        $this->printLine();
        $this->Ln();
    }


    /**
     * Returns the params for the given mask format.
     *
     * Returns the params for the given mask format, treating the defaults that were not override.
     *
     * @param string $customFormat The mask format to get the params.
     * @return array The params to configure the current mask.
     */
    private function getParams($customFormat){
        $params = [];
        foreach (($this->masks[$customFormat['type']] ?? []) as $param => $value) {
            if(isset($customFormat['params'][$param])){
                $params[$param] = $customFormat['params'][$param];
            }
            else {
                $params[$param] = $value;
            }
        }
        return $params;
    }

    /**
     * Apply number mask on the given value.
     *
     * Tests if the mask allows negative and zero and, if yes return the floatval of the given number.
     * If not test if the number is according to the params and returns the floatval if yes. If not
     * returns "".
     *
     * @param mixed $value  The value to apply the mask.
     * @param array $params The masks params.
     *
     * @return mixed The floatval of the params if the value pass the params. Empty string if does
     * not pass
     */
    public function number($value, $params){
        $value = strlen($value) ? floatval($value) : null;

        if((!$params['allowNegative'] && $value < 0) ||
            (!$params['allowZero'] && $value == 0)){
            return "";
        }
        return $value;
    }

    public function zerofill($value, $params){
        $padType = !empty($params['right']) ? STR_PAD_RIGHT : STR_PAD_LEFT;
        $length = 2;
        if(!empty($params['maxlength'])) {
            $length = $params['maxlength'];
        }
        if(strlen($value) < $length) {
            $value = str_pad($value, $length, '0', $padType);
        }
        return $value;
    }

    /**
     * Translate the given format to the used on the report.
     *
     * Translate the format passed (in moment.js structure) to the php datetime format using the
     * dictionary ($momentToPhpDateDictionary).
     *
     * @param string $format Format to be translated (in moment.js format).
     * @return string Format translated to the used on datetime.
     */
    private function translateDateFormat($format){
        return strtr($format, $this->momentToPhpDateDictionary);
    }

    public function datetime($value, $params){
        return $this->date($value, $params);
    }

    /**
     * Apply date mask on the given value.
     *
     * Instantiate a Datetime with the given value, translate the given format on params and return
     * the value on the desired format.
     *
     * @param string $value Date to be formatted.
     * @param array $params Params to translate the given date.
     *
     * @return string Date formatted.
     */
    public function date($value, $params){
        if(!empty($value)) {
            $format = $this->translateDateFormat($params['format']);
            if(strpos($format, ' ') === false) {
                $value = explode(' ', $value)[0];
            }
            $d = \DateTime::createFromFormat($format, $value);
            return $d->format($format);
        }
        return $value;
    }

    /**
     * Apply time mask on the given value.
     *
     * @param int    $value  The time as integer.
     * @param array  $params Params to translate the given time.
     *
     * @return string Time formatted.
     */
    public function time($value, $params){
        $time = '';
        $withSeconds = $params['withSeconds'];
        $dateTime = new \DateTime;
        $dateTime->setTimestamp($value);
        $date = getDate($dateTime->getTimestamp());

        $seconds = $date['seconds'];
        $minutes = $date['minutes'];
        $hours = $date['hours'];

        if ($hours > 0) $time .= $hours . 'H';
        if ($minutes > 0) $time .= $minutes . 'm';
        if ($withSeconds && $seconds > 0) $time .= $seconds . 's';

        return $time;
    }

    /**
     * Apply float mask on the given value.
     *
     * Apply the number mask on the given value and returns the number_format with the given params.
     *
     * @param mixed $value Value to be formatted.
     * @param array $params Params to format the given value.
     *
     * @return mixed The value formatted or a empty string if the value does not pass the mask params.
     */
    public function float($value, $params){
        $value = $this->number($value, $params);

        if(!is_null($value)){
            $value = number_format($value, $params['precision'], $params['decimal'], $params['thousands']);
        }
        return $value;
    }

    public function viewFloat($value, $params){
        $value = $this->number($value, $params);

        if(!is_null($value)){
            $value = number_format($value, $params['precision'], $params['decimal'], $params['thousands']);
        }
        return $value;
    }

    /**
     * Apply fix mask on the given value.
     *
     * Apply the number mask on the given value and returns the number_format with the given params.
     *
     * @param string $value Value to be formatted.
     * @param array $params Params to format the given value.
     *
     * @return string The value formatted or a empty string if the value does not pass the mask params.
     */
    public function fix($value, $params){
        $value = "$value";
        $patternsTranslation = [
            "0" => [
                "pattern" => "/\d/"
            ],
            "9" => [
                "pattern" => "/\d/"
            ],
            "#" => [
                "pattern" => "/\d/"
            ],
            "A" => [
                "pattern" => "/[a-zA-Z0-9]/"
            ],
            "*" => [
                "pattern" => "/[a-zA-Z0-9]/"
            ],
            "S" => [
                "pattern" => "/[a-zA-Z]/"
            ],
            "a" => [
                "pattern" => "/[a-zA-Z]/"
            ]
        ];
        $maskPattern = $params["mask"];
        $parsedValue = "";

        if(strlen($value) && strlen($maskPattern)) {
            $j = 0;
            for ($i = 0; $i < strlen($maskPattern); $i++) {
                $currentDigit = isset($value[$i]) ? $value[$i] : "";
                $patternDigit = isset($maskPattern[$j]) ? $maskPattern[$j] : "";
                $translation = isset($patternsTranslation[$patternDigit]) ?
                    $patternsTranslation[$patternDigit] : null;

                if(is_null($translation)){
                    if($translation !== $currentDigit){
                        if(!is_null($patternDigit)) {
                            $parsedValue .= $patternDigit;
                        }
                        $parsedValue .= $currentDigit;
                        $j++;
                    }
                } elseif(preg_match($translation['pattern'], $currentDigit)) {
                    $parsedValue .= $currentDigit;
                }

                $j++;
            }
        } else {
            $parsedValue = $value;
        }

        return $parsedValue;
    }

    /**
     * Apply currency mask on the given value.
     *
     * Apply the float mask on the value and returns it with the symbol if the params are setted to do
     * it.
     *
     * @param string $value Value to be formatted.
     * @param array $params Params to format the given value.
     *
     * @return string The value formatted or a empty string if the value does not pass the mask params.
     */
    public function currency($value, $params){
        $value = $this->float($value, $params);
        if($params['showSymbol']) {
            $value = "$params[symbol] $value";
        }
        return $value;
    }

    /**
     * Apply percent mask on the given value.
     *
     * Apply the percent mask on the value calculating the percentage based on the min and max
     * values.
     *
     * @param mixed $value Value to be formatted.
     * @param [type] $params Params to format the given value.
     * @return string The value formatted.
     */
    public function percent($value, $params){
        $value = floatval($value);

        $min = $params['minValue'];
        $max = $params['maxValue'] - $min;
        $value -= $min;

        $value = 100 * $value / $max;
        $value = $this->float($value, $params)."%";
        return $value;
    }

    /**
     * Format the given value with the column's masks.
     *
     * Tests if there is a format to configured the value and if there is get the column's params
     * and returns the call of the function with the exact name of the column's mask.
     *
     * @param mixed $value  The value to be translated.
     * @param array $column The column configured with the desired format.
     *
     * @return mixed The given value formatted.
     */
    private function formatValue($value, $column){
        if (!isset($column['format'])) return $value;
        if($params = $this->getParams($column['format'])) {
            $functionName = $column['format']['type'];
            $value = $this->$functionName($value, $params);
        }
        return $value;
    }

    /**
     * Tests if the value is to be formatted.
     *
     * Tests if with the given state the value should be formatted.
     *
     * @param bool $valueIsFormatted Tell if the value should be formatted.
     * @param array $row Row to be printed.
     * @param array $column Column metadata.
     * @param string $columnName The name of the given column metadata.
     *
     * @return boolean Result telling if is to format the given value.
     */
    private function isToFormat($valueIsFormatted, $column, $expressions = null, $mask = null){
        return  !($valueIsFormatted && ((isset($column['format']) && isset($column['format']['type']) &&
                ($column['format']['type'] == 'date' || $column['format']['type'] == 'datetime' || $column['format']['type'] == 'time')) &&
                (is_null($expressions) || $expressions !== 'COUNT') &&
                (is_null($mask) || strtolower($mask) !== 'fix')));
    }

    /**
     * Print the given row.
     *
     * Config the font to 'common', set the correct x axis, format the value if it's not and print
     * it on the report.
     *
     * @param array $row Row to be printed.
     * @param bool $valueIsFormatted Indicates if the given value is already formatted.
     */
    private function printRow($row, $valueIsFormatted=false){
        $this->configFont('common');
        $this->SetX($this->GetX() + (sizeof($this->groups)) * self::TAB_SIZE);
        $this->SetFillColor(255);
        foreach($this->columns as $columnName => $column){
            if(empty($this->groups[$columnName])) {
                if($this->isToFormat($valueIsFormatted, $column)){
                    $value = $this->formatValue(isset($row[$columnName]) ? $row[$columnName] : "", $column);
                }
                else{
                    if(isset($row[$columnName]) && !is_null($row[$columnName])){
                        $value = $row[$columnName];
                    }
                    else {
                        $value = "";
                    }
                }
                if(is_string($value)){
                    $value = $this->internationalize($value);
                }

                $this->Cell(
                    $column['size'],
                    5, $value, 0, 0,
                    $column['align'],
                    true
                );
            }
        }
        $this->Ln();
    }

    /**
     * Get the size of the group print.
     *
     * Calculate the height of the row's group impression based on the amount of the groups and
     * if they have expressions.
     *
     * @param array $row The row to have it's groups print size calculated.
     *
     * @return int The height of the groups print.
     */
    private function getPrintGroupSize($row){
        $size = $this->textSize*5;
        foreach($this->groupNames as $key){
            if(isset($row[$key])){
                $size += 5;
            }
        }

        return $size;
    }

    /**
     * Finishes the page and add a new one.
     *
     * Tests if a totalizator should be printed after the last row, if yes, do it, and adds a new page.
     *
     * @param array $row The row to be tested.
     */
    private function finishPage($row ,$valueIsFormatted=false){
        foreach($this->groupValues as $key => $group){
            if(!$valueIsFormatted && isset($this->groupValues[$key]['value']) && strcmp($this->groupValues[$key]['value'], $row[$key])!==0){
                $this->clearOncomingGroups($key);
            }
        }
        $this->AddPage();
    }

    /**
     * Tests if the row print will overflow.
     *
     * Tests if the page height is smaller than the current y axis plus the footer height plus a blank space height plus the group
     * size print.
     *
     * @param array $row                The row to be tested.
     * @param bool  $valueIfFormatted   Defines if the value is formatted.
     * @return bool The test if the row being printed will overflow the page.
     */
    private function testHeightOverflow($row, $valueIsFormatted=false){
        if($valueIsFormatted){
            $groupsAndHeadersSize = ($this->indentLevel + 1);
            if($this->hasExpression) $groupsAndHeadersSize = $groupsAndHeadersSize*2;
            return $this->GetPageHeight() < self::FOOTER_HEIGHT($this->isWebService) + self::BLANK_SPACE($this->isWebService) +  $this->GetY() + $groupsAndHeadersSize;
        } else {
            return $this->GetPageHeight() < self::FOOTER_HEIGHT($this->isWebService) +  $this->GetY() + self::BLANK_SPACE($this->isWebService) + $this->getPrintGroupSize($row);
        }
    }

    public function writeRows(array $rows, $valueIsFormatted=false){
        if($valueIsFormatted){
            $lastGroupHeader = null;
            $groupHeaders = array();
            $waitForNextRow = false;
            foreach($rows as $row){
                if($this->testHeightOverflow($row, true)){
                    $waitForNextRow = true;
                    $this->indentLevel = 0;
                }
                if(isset($row['__groupHeader']) && $row['__groupHeader']){
                    if(!$waitForNextRow) {
                        if(!$this->hasExpression) {
                            if($lastGroupHeader === $row['__groupLabel']){
                                $this->indentLevel--;
                            } elseif(in_array($row['__groupLabel'], $groupHeaders)) {
                                $key = array_search($row['__groupLabel'], $groupHeaders);
                                $groupHeaders = array_slice($groupHeaders, 0, $key);
                                $this->indentLevel = count($groupHeaders);
                            }
                        }

                        $this->printGroupHeader($row, null);
                        $groupHeaders[] = $row['__groupLabel'];
                        $lastGroupHeader = $row['__groupLabel'];
                        $this->indentLevel++;
                    }
                } elseif(isset($row['__groupFooter']) && $row['__groupFooter']) {
                    if($this->indentLevel > 0) $this->indentLevel--;
                    $this->printRowAsFooter($row);
                } else {
                    if($waitForNextRow){
                        $this->finishPage($row, true);
                        $waitForNextRow = false;

                        foreach($this->groupValues as $key => $group){
                            $this->printGroupHeader($row, $key);
                            $this->groupValues[$key]['value'] = $row[$key];
                            $this->indentLevel++;
                        }
                    }

                    $this->printRow($row, true);
                }
            }
        } else {
            foreach($rows as $row){
                if($rePrintGroups = $this->testHeightOverflow($row)){
                    $this->finishPage($row, false);
                }
                $this->printGroups($row, $rePrintGroups, false);
                $this->printRow($row, false);
            }
        }
    }

    private function printRowAsFooter($row){
        $this->configFont('bCommon');

        if (isset($row['__groupLabel']) && (isset($row['__groupValue']))) {
            $rowValue = "Total - ";

            if(empty($row['__groupLabel'])){
                $rowValue .= "{$row['__groupValue']}";
            } else {
                $rowValue .= "{$row['__groupLabel']}: {$row['__groupValue']}";
            }
        } else {
            $rowValue = "TOTAL:";
        }
        $rowValue = $this->internationalize($rowValue);

        $this->SetX((self::TAB_SIZE*$this->indentLevel) + $this->lMargin);
        $this->Cell(4, 5, $rowValue, 0, 0);
        $this->Ln();

        $this->SetX(count($this->groups) * self::TAB_SIZE + $this->lMargin);

        foreach($this->columns as $columnName => $column){
            $value = isset($row[$columnName]) ? $row[$columnName] : "";

            $this->Cell(
                $column['size'],
                5, $value, 0, 0,
                $column['align']
            );
        }

        $this->printLine();
        $this->Ln();
    }

    /**
     * Clear the white margin.
     *
     * Print a white block on the right side of the page to prevent any row's value out of the page.
     */
    private function clearRightMargin(){
        $this->SetFillColor(255);
        $x = $this->GetPageWidth() - $this->rMargin;
        $width = $this->rMargin;
        $y = 0;
        $height = $this->GetPageHeight();

        $this->Rect($x, $y, $width, $height, "F");
    }

    /**
     * Print the image on the footer.
     *
     * Print the image inside the footer of the report.
     *
     * @param string $image  The image name.
     */
    private function printFooterImage($image){
        if(isset($this->metaData[$image])){
            $pic = $this->metaData[$image];

            $dataPieces = explode(',', $pic);
            $type = explode( ';' ,explode('/', $dataPieces[0])[1])[0];
            $this->Image($pic, null, null, 15, 5, $type);
        }
    }

    /**
     * Print the report footer.
     *
     * Print the footer's line, config the font to 'footer', print the report title and it's version.
     * Also clear the right margin.
     */
    public function Footer(){
        $this->SetY(-20);
        $this->configFont('footer');

        $this->Ln();
        $word = $this->translate('Report');
        $title = $this->internationalize($this->metaData['title']);
        $this->Cell(0, 14, "$word: ".$title, 0, 0, 'R');

        $y = ($this->getY()) + 1;

        $this->Cell(15, 0, "", 0, 0);
        $this->setXY(65 + $this->lMargin , $y);
        $this->printFooterImage('reportEnterpriseLogo');

        $this->Cell(15, 0, "", 0, 0);
        $this->setXY( -(80+$this->rMargin) , $y);
        $this->printFooterImage('reportProductLogo');

        $this->Ln();
        $word = $this->translate('Version');
        $this->Cell(0, 16, "$word: ".$this->metaData['version'], 0, 0, 'R');

        $this->clearRightMargin();
    }

    /**
     * Prints the footer of the last row.
     *
     * Prints the footer of the last row.
     *
     */
    private function printLastTotalizator($valueIsFormatted=false){
        reset($this->groupValues);
        if(!$valueIsFormatted) $this->clearOncomingGroups(key($this->groupValues));
    }

    /**
     * Prints the global footer total.
     *
     * Prints the global footer total.
     *
     */
    private function printGlobalTotal($currentColumn, $valueIsFormatted=false){
        $this->configFont('bCommon');
        $rowValue = "TOTAL: ";
        $this->SetX(self::TAB_SIZE + $this->lMargin);
        $this->Cell(4, 5, $rowValue, 0, 0);
        $this->Ln();

        $this->SetX(count($this->groups) * self::TAB_SIZE + $this->lMargin);
        foreach($this->columns as $columnName => $column){
            if(!isset($column["expression"]))
                $value = "";
            else {
                $mask = null;
                if(isset($column['format']) && isset($column['format']['type'])){
                    $mask = $column['format']['type'];
                }

                $value = $this->totalValues[$columnName][$currentColumn];

                if($this->isToFormat($valueIsFormatted, $column, $column['expression'], $mask)){
                    $value = $this->formatValue($value, $column);
                }
            }
            if($column['size'] != 0 && !in_array($columnName, $this->groupNames)) {
                $this->Cell(
                    $column['size'],
                    5, $value, 0, 0,
                    $column['align']
                );
            }
        }
        $this->printLine();
        $this->Ln();
    }

        /**
     * Prints the last subtitle with a custom title.
     *
     * This method sets the font configuration to 'bCommonGreater', internationalizes the custom title,
     * and then prints it as the last subtitle on the page.
     *
     * @param string $customTitle The custom title to be printed.
     */
    private function printLastSubTitle($customTitle){
            $this->configFont('bCommonGreater');
            $title = $this->internationalize($customTitle);
            $this->Cell(0, 10, $title, 0, 1, 'C');
    }

    /**
     * Adds the last page to the document and prints the applied filters.
     *
     * This function marks the document as the last page, adds a new page to the document,
     * and prints the applied filters on the last page.
     */
    public function lastPage(){
        $this->thisLastPage = true;
        if(empty($this->metaData['filter'])){
            $this->AddPage();
            $this->SetY(21.5);
            $this->printLine();
            $this->breakFilterLine();

            $this->printLastSubTitle($this->getTranslations('no_filter', $this->metaData['lang']));
        }
        else {
            $this->AddPage();
            $this->printFilter();
        }
    }

     /**
     * Output the file to the given target.
     *
     * Output the file on the "F" on the given target file.
     *
     * @param string $targetFile Full path to save the file.
     */

    public function finish($targetFile, $valueIsFormatted=false){
        if(!$valueIsFormatted){
            $this->printLastTotalizator();

            if(count($this->totalValues)){
                $this->printGlobalTotal(end($this->groupNames), $valueIsFormatted);
            }
        }
        $this->lastPage();
        $this->Output("F", $targetFile);
    }

    /**
     * Return the format of the file.
     *
     * Return the format of the generate file.
     *
     * @return string The report file format.
     */
    public function formatName(){
        return "PDF";
    }
}
