<?php

namespace Zeedhi\ZhuAcl\Service;

use Doctrine\ORM\EntityManager;
use Zeedhi\ZhuAcl\Util\Environment;
use Zeedhi\ZhuAcl\Util\TableProperties;
use Zeedhi\ZhuAcl\Model\Repositories\ZhuAccessControlElement;
use Zeedhi\ZhuAcl\Model\Repositories\ZhuAccessControlRule;
use Zeedhi\ZhuAcl\Model\Entities\ZhuAccessControlRule as ZhuACLRuleEntity;
use Zeedhi\ZhuAcl\Model\Entities\ZhuAccessControlElement as ZhuACLElementEntity;

class AccessControlRule {

    /** @var EntityManager */
    protected $zhuEntityManager;
    /** @var Environment */
    protected $environment;

    /**
     * @param EntityManager
     * @param Environment
     */
    public function __construct(EntityManager $zhuEntityManager, Environment $environment){
        $this->zhuEntityManager = $zhuEntityManager;
        $this->environment = $environment;
    }

    /**
     * @param $row
     * @return null|ZhuACLRuleEntity
     */
    protected function findElementRule($row){
        if(isset($row[TableProperties::COLUMN_RULE_ID])){
            return $this->zhuEntityManager->find(ZhuACLRuleEntity::class, $row[TableProperties::COLUMN_RULE_ID]);
        }
    }

    /**
     * @param $row
     * @return null|string
     */
    public function deleteRule($row){
        $rule = $this->findElementRule($row);
        if($rule){
            $this->zhuEntityManager->remove($rule);
            $this->zhuEntityManager->flush();
            return $row[TableProperties::COLUMN_UNIQUE_ID];
        }
    }

    protected function deleteAllProfileRules($orgId, $profileId){
        $queryBuilder = $this->zhuEntityManager->getRepository(ZhuACLRuleEntity::class)->createQueryBuilder('rule')
            ->delete()
            ->where('rule.groupId = :groupId and rule.organizationId = :orgId')
            ->setParameters(array('groupId' => $profileId, 'orgId' => $orgId));
        $queryBuilder->getQuery()->execute();
    }

    protected function cloneRules($orgId, $profileFrom, $profileTo){
        $ruleFilter = array(
            "organizationId" => $orgId,
            "groupId" => $profileFrom
        );
        $rules = $this->zhuEntityManager->getRepository(ZhuACLRuleEntity::class)->findBy($ruleFilter);
        foreach($rules as $rule){
            $clonedRule = clone $rule;
            $clonedRule->setId(null);
            $clonedRule->setGroupId($profileTo);
            $this->zhuEntityManager->persist($clonedRule);
        }
    }

    public function copyRules($profileFrom, $profileTo){
        $orgId = $this->environment->getOrganizationId();
        try {
            $this->zhuEntityManager->beginTransaction();
            $this->deleteAllProfileRules($orgId, $profileTo);
            $this->cloneRules($orgId, $profileFrom, $profileTo);
            $this->zhuEntityManager->flush();
            $this->zhuEntityManager->commit();
        } catch(\Exception $e){
            $this->zhuEntityManager->rollback();
            throw new \Exception($e->getMessage(), 1);
        }
    }

    /**
     * @param ZhuACLElementEntity $element
     * @param ZhuACLRuleEntity $rule
     * @throws Exception
     */
    protected function persistElementAndRule($element, $rule){
        try {
            $this->zhuEntityManager->persist($element);
            $this->zhuEntityManager->persist($rule);
            $this->zhuEntityManager->flush();
            $this->zhuEntityManager->commit();
        } catch(\Exception $e){
            $this->zhuEntityManager->rollback();
            throw new \Exception($e->getMessage(), 1);
        }
    }

    /**
     * @param $row
     * @param ZhuACLElementEntity $element
     * @return ZhuACLRuleEntity
     */
    protected function createNewRule($row, ZhuACLElementEntity $element){
        $rule = new ZhuACLRuleEntity();
        $rule->setCreatedBy($row[TableProperties::COLUMN_USER_ID]);
        $rule->setOrganizationId($row[TableProperties::COLUMN_ORGANIZATION_ID]);
        $rule->setAccessControlElement($element);
        $rule->setName($row[TableProperties::COLUMN_NAME]);
        $rule->setGroupId($row[TableProperties::COLUMN_GROUP_ID]);
        return $rule;
    }

    /**
     * @param $row
     * @param ZhuACLElementEntity $element
     * @return ZhuACLRuleEntity
     */
    protected function getRule($row, ZhuACLElementEntity $element){
        $ruleFilter = array(
            "organizationId" => $row[TableProperties::COLUMN_ORGANIZATION_ID],
            "groupId" => $row[TableProperties::COLUMN_GROUP_ID],
            "accessControlElement" => $element->getId()
        );
        $rule = $this->zhuEntityManager->getRepository(ZhuACLRuleEntity::class)->findOneBy($ruleFilter);
        if(!isset($rule)){
            $rule = $this->createNewRule($row, $element);
        }
        $rule->setUpdatedBy($row[TableProperties::COLUMN_USER_ID]);
        $rule->setRule($row[TableProperties::COLUMN_RULE]);
        $rule->setUpdated(new \DateTime());
        return $rule;
    }

    /**
     * @param $row
     * @return ZhuACLElementEntity
     */
    protected function createNewElement($row){
        $element = new ZhuACLElementEntity();
        $element->setCreatedBy($row[TableProperties::COLUMN_USER_ID]);
        $element->setType($row[TableProperties::COLUMN_TYPE]);
        $element->setUniqueId($row[TableProperties::COLUMN_UNIQUE_ID]);
        $element->setProjectId($row[TableProperties::COLUMN_PROJECT_ID]);
        return $element;
    }

    /**
     * @param $row
     * @return ZhuACLElementEntity
     */
    protected function getElement($row){
        $elementFilter = array(
            "uniqueId" => $row[TableProperties::COLUMN_UNIQUE_ID],
            "projectId" => $row[TableProperties::COLUMN_PROJECT_ID]
        );
        $element = $this->zhuEntityManager->getRepository(ZhuACLElementEntity::class)->findOneBy($elementFilter);
        if(!isset($element)){
            $element = $this->createNewElement($row);
        }
        $element->setUpdatedBy($row[TableProperties::COLUMN_USER_ID]);
        $element->setName($row[TableProperties::COLUMN_NAME]);
        $element->setLabel($row[TableProperties::COLUMN_LABEL]);
        $element->setUpdated(new \DateTime());
        return $element;
    }

    /**
     * @param $row
     * @return ZhuACLRuleEntity
     */
    public function insertRule($row){
        $row[TableProperties::COLUMN_PROJECT_ID] = $this->environment->getProjectId();
        $row[TableProperties::COLUMN_USER_ID] = $this->environment->getUserId();
        $row[TableProperties::COLUMN_ORGANIZATION_ID] = $this->environment->getOrganizationId();

        $this->zhuEntityManager->beginTransaction();
        $element = $this->getElement($row);
        $rule = $this->getRule($row, $element);

        $this->persistElementAndRule($element, $rule);
        return $rule;
    }
}