<?php

namespace Zeedhi\ZhuAcl\Controller;

use Zeedhi\ZhuAcl\Util\Environment;
use Zeedhi\ZhuAcl\Util\TableProperties;
use Zeedhi\ZhuAcl\Service\AccessControlRule as AccessControlRuleService;
use Zeedhi\ZhuAcl\Model\Repositories\ZhuAccessControlElement;
use Zeedhi\Framework\Controller\Crud;
use Zeedhi\Framework\DataSource\FilterCriteria;
use Zeedhi\Framework\DataSource\Manager;
use Zeedhi\Framework\DataSource\ParameterBag;
use Zeedhi\Framework\DataSource\DataSet;
use Zeedhi\Framework\DTO;
use Zeedhi\Framework\DTO\Response\Notification;

class AccessControlRule extends Crud {

    /** @var ZhuAccessControlElement */
    protected $zhuAccessControlElementRepository;

    /** @var ParameterBag */
    protected $parameterBag;

    /** @var AccessControlRuleService */
    protected $accessControlRuleService;

    /** @var Environment */
    protected $environment;

    protected $dataSourceName = "aclZhuAccessControlRule";

    protected $parameterBagColumns = array(TableProperties::COLUMN_PROJECT_ID, TableProperties::COLUMN_ORGANIZATION_ID);

    public function __construct(
        Manager $dataSourceManager, ParameterBag $parameterBag, AccessControlRuleService $accessControlRuleService,
        Environment $environment
    ) {

        parent::__construct($dataSourceManager);
        $this->parameterBag = $parameterBag;
        $this->accessControlRuleService = $accessControlRuleService;
        $this->environment = $environment;
    }

    /**
     * @param $row
     * @return $containerRow
     */
    protected function prepareContainerRow($row){
        $containerRow = array(
            TableProperties::COLUMN_TYPE => TableProperties::TYPE_CONTAINER,
            TableProperties::COLUMN_UNIQUE_ID => $row[TableProperties::COLUMN_CONTAINER_ID],
            TableProperties::COLUMN_GROUP_ID => $row[TableProperties::COLUMN_GROUP_ID],
            TableProperties::COLUMN_NAME => $row[TableProperties::COLUMN_CONTAINER_NAME],
            TableProperties::COLUMN_LABEL => $row[TableProperties::COLUMN_CONTAINER_LABEL],
            TableProperties::COLUMN_PROJECT_ID => $row[TableProperties::COLUMN_PROJECT_ID],
            TableProperties::COLUMN_RULE => $row[TableProperties::COLUMN_RULE]
        );

        if(isset($row[TableProperties::COLUMN_CONTAINER_RULE_ID])){
            $containerRow[TableProperties::COLUMN_RULE_ID] = $row[TableProperties::COLUMN_CONTAINER_RULE_ID];
        }

        return $containerRow;
    }

    protected function saveContainerRule($row){
        $containerRow = $this->prepareContainerRow($row);
        $containerRule = $this->accessControlRuleService->insertRule($containerRow);
        return $containerRule;
    }

    protected function saveParentMenuRule($row){
        $parentMenuRow = array(
            TableProperties::COLUMN_TYPE => TableProperties::TYPE_MENU,
            TableProperties::COLUMN_UNIQUE_ID => $row[TableProperties::COLUMN_PARENT_MENU_ID],
            TableProperties::COLUMN_GROUP_ID => $row[TableProperties::COLUMN_GROUP_ID],
            TableProperties::COLUMN_NAME => $row[TableProperties::COLUMN_PARENT_MENU_NAME],
            TableProperties::COLUMN_LABEL => $row[TableProperties::COLUMN_PARENT_MENU_LABEL],
            TableProperties::COLUMN_PROJECT_ID => $row[TableProperties::COLUMN_PROJECT_ID],
            TableProperties::COLUMN_RULE => $row[TableProperties::COLUMN_RULE]
        );

        if(isset($row[TableProperties::COLUMN_PARENT_MENU_RULE_ID])){
            $parentMenuRow[TableProperties::COLUMN_RULE_ID] = $row[TableProperties::COLUMN_PARENT_MENU_RULE_ID];
        }
        $parentMenuRule = $this->accessControlRuleService->insertRule($parentMenuRow);
        return $parentMenuRule;
    }

    public function save(DTO\Request\DataSet $request, DTO\Response $response) {
        $updatedRules = array();
        $deletedRules = array();

        foreach($request->getDataSet()->getRows() as $row) {
            $uniqueId = $row[TableProperties::COLUMN_UNIQUE_ID];
            $hasContainerRule = isset($row[TableProperties::COLUMN_CONTAINER_ID]) && $row[TableProperties::COLUMN_TYPE] == TableProperties::TYPE_MENU;

            if(isset($row[TableProperties::COLUMN_PARENT_MENU_ID])){
                $this->saveParentMenuRule($row);
            }

            if(isset($row['__isDeleted']) && $row['__isDeleted']){
                $ruleId = $this->accessControlRuleService->deleteRule($row);
                if($ruleId){
                    array_push($deletedRules, $ruleId);
                }

                if($hasContainerRule && isset($row['RULE_SAVE_CONTAINER'])){
                    $row[TableProperties::COLUMN_RULE] = $row['RULE_SAVE_CONTAINER'];
                    $containerRule = $this->saveContainerRule($row);
                    $updatedRules[$uniqueId] = array(
                        "menuId" => $ruleId,
                        "containerId" => $containerRule->getId()
                    );
                }
            } else {
                $rule = $this->accessControlRuleService->insertRule($row);
                $ruleId = $rule->getId();

                if($hasContainerRule){
                    $containerRule = $this->saveContainerRule($row);
                    $updatedRules[$uniqueId] = array(
                        "menuId" => $ruleId,
                        "containerId" => $containerRule->getId()
                    );
                } else {
                    $updatedRules[$uniqueId] = $ruleId;
                }
            }
        }

        $modifiedRules = array("deleted"=> $deletedRules, "updated"=> $updatedRules);

        $rulesDataSet = new DataSet($this->dataSourceName, $modifiedRules);
        $response->addDataSet($rulesDataSet);
    }

    public function exportRules(DTO\Request\DataSet $request, DTO\Response $response) {
        $row = $request->getDataSet()->getRows()[0];
        $profileFrom = $row['profileFrom'];
        try {
            foreach($row['profilesTo'] as $profileTo) {
                $this->accessControlRuleService->copyRules($profileFrom, $profileTo);
            }
            $response->addNotification(new Notification("Regras exportadas com sucesso!", Notification::TYPE_SUCCESS));
        } catch (\Exception $e){
            $response->addNotification(new Notification("Não foi possível exportar as regras.", Notification::TYPE_ERROR));
        }
    }

    public function find(DTO\Request\Filter $request, DTO\Response $response) {
        $this->parameterBag->set('ORGANIZATION_ID', $this->environment->getOrganizationId());
        $this->parameterBag->set('PROJECT_ID', $this->environment->getProjectId());

        $filterCriteria = $request->getFilterCriteria();
        $newFilterCriteria = new FilterCriteria(
            $this->dataSourceName,
            $filterCriteria->getConditions(),
            $filterCriteria->getPage(),
            $filterCriteria->getPageSize()
        );

        $dataSet = $this->dataSourceManager->findBy($newFilterCriteria);
        $response->addDataSet($dataSet);
    }
}